import 'package:cached_network_image/cached_network_image.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:road24_mobile/data/default_image.dart';
import 'package:road24_mobile/data/response/CarsResponse.dart' as CR;
import 'package:road24_mobile/model/utils/styles/AppStyles.dart';
import 'car_number.dart';

class CarsListItem extends StatelessWidget {
  final CR.Results carDetails;
  final GestureTapCallback onTapEdit;

  const CarsListItem({required this.carDetails, required this.onTapEdit});

  @override
  Widget build(BuildContext context) {
    if (carDetails.image?.file != null)
      return _widgetWithImage(context, modelImg: carDetails.image!.file!);
    return _widgetNoImage(
      context,
      brandImg: carDetails.carBrand?.logo?.file ??
          DefaultImages.brandImage(carDetails.brand)?.file,
      modelImg: carDetails.carModel?.image?.file ??
          DefaultImages.modelImage(carDetails.model)?.file,
    );
  }

  Widget _widgetWithImage(
    BuildContext context, {
    double ratioX = 4,
    double ratioY = 3,
    required String modelImg,
  }) {
    final double width = 320;
    final double height = width * ratioY / ratioX;

    return ClipRRect(
      borderRadius: BorderRadius.circular(16),
      child: Stack(
        fit: StackFit.passthrough,
        children: [
          CachedNetworkImage(
            width: width,
            height: height,
            imageUrl: modelImg,
            fit: BoxFit.fitWidth,
            placeholder: (_, __) => Image.asset(
              "assets/brand/placeholder.jpeg",
              width: width,
              height: height,
              fit: BoxFit.fitWidth,
            ),
            errorWidget: (_, __, ___) => Image.asset(
              "assets/png/covered_car.png",
              width: width,
              height: height,
              fit: BoxFit.fitWidth,
            ),
          ),
          Container(
            width: width,
            height: height,
            padding: const EdgeInsets.all(14),
            color: Colors.black.withOpacity(0.3),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Row(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Expanded(
                      child: Text(
                        '${carDetails.brand} ${carDetails.model}',
                        maxLines: 2,
                        overflow: TextOverflow.ellipsis,
                        style: AppStyle.textStyle().copyWith(
                          color: Colors.white,
                          fontSize: 20,
                          fontWeight: FontWeight.w600,
                        ),
                      ),
                    ),
                    const SizedBox(width: 20),
                    GestureDetector(
                      behavior: HitTestBehavior.opaque,
                      onTap: onTapEdit,
                      child: Container(
                        width: 40,
                        height: 40,
                        decoration: BoxDecoration(
                          color: Color(0xffECECEC),
                          shape: BoxShape.circle,
                        ),
                        child: const Icon(Icons.edit_outlined,
                            color: Color(0xff282828)),
                      ),
                    ),
                  ],
                ),
                const Spacer(),
                CarNumber(carNumber: carDetails.number, fontSize: 18),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _widgetNoImage(
    BuildContext context, {
    double ratioX = 4,
    double ratioY = 3,
    String? brandImg,
    String? modelImg,
  }) {
    final double width = 320;
    final double height = width * ratioY / ratioX;
    return Container(
      width: width,
      height: height,
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(16.0),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              ClipRRect(
                borderRadius: BorderRadius.circular(12),
                child: CachedNetworkImage(
                  imageUrl: "$brandImg",
                  width: 40,
                  height: 40,
                  fit: BoxFit.fill,
                  errorWidget: (context, url, error) {
                    return Image.asset(
                      "assets/brand/placeholder.jpeg",
                      width: 40,
                      height: 40,
                      fit: BoxFit.fill,
                    );
                  },
                ),
              ),
              const SizedBox(width: 4),
              Expanded(
                flex: 2,
                child: SizedBox(
                  height: 40,
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        carDetails.brand,
                        style: AppStyle.textStyle().copyWith(
                            color: Color(0xff747474),
                            fontWeight: FontWeight.normal,
                            fontSize: 14.0),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                      const Spacer(),
                      Text(
                        carDetails.model,
                        style: AppStyle.textStyle().copyWith(
                            color: Color(0xff747474),
                            fontWeight: FontWeight.normal,
                            fontSize: 14.0),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ],
                  ),
                ),
              ),
              const SizedBox(width: 4),
              GestureDetector(
                behavior: HitTestBehavior.opaque,
                onTap: onTapEdit,
                child: Container(
                  width: 40,
                  height: 40,
                  decoration: BoxDecoration(
                    color: Color(0xffECECEC),
                    shape: BoxShape.circle,
                  ),
                  child:
                      const Icon(Icons.edit_outlined, color: Color(0xff282828)),
                ),
              ),
            ],
          ),
          Expanded(
            child: CachedNetworkImage(
              imageUrl: "$modelImg",
              width: double.infinity,
              fit: BoxFit.fitHeight,
              errorWidget: (context, url, error) {
                return Image.asset(
                  "assets/png/covered_car.png",
                  fit: BoxFit.fitWidth,
                );
              },
            ),
          ),
          Container(
            width: double.infinity,
            alignment: Alignment.center,
            child: CarNumber(carNumber: carDetails.number),
          ),
        ],
      ),
    );
  }
}
