import 'package:cached_network_image/cached_network_image.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter/painting.dart';
import 'package:road24_mobile/data/default_image.dart';
import 'package:road24_mobile/data/response/CarsResponse.dart' as CR;
import 'package:road24_mobile/model/utils/styles/AppColors.dart';
import 'package:road24_mobile/model/utils/styles/AppStyles.dart';
import 'package:road24_mobile/view/built/i18n.dart';
import 'package:road24_mobile/view/widgets/items/car_number.dart';
import 'package:shimmer/shimmer.dart';

class CarsListItem extends StatelessWidget {
  final CR.Results? carDetails;
  final GestureTapCallback onTapItem;
  final GestureTapCallback onTapPinned;
  final bool isPinned;

  const CarsListItem({
    required this.carDetails,
    required this.onTapItem,
    required this.onTapPinned,
    required this.isPinned,
  });

  @override
  Widget build(BuildContext context) {
    if (carDetails == null) return _widgetShimmer(context);
    if (carDetails?.image?.file != null)
      return _widgetWithImage(
        context,
        width: 280,
        modelImg: carDetails!.image!.file!,
      );
    return _widgetNoImage(
      context,
      width: 280,
      brandImg: carDetails?.carBrand?.logo?.file ??
          DefaultImages.brandImage(carDetails!.brand)?.file,
      modelImg: carDetails?.carModel?.image?.file ??
          DefaultImages.modelImage(carDetails!.model)?.file,
    );
  }

  Widget _widgetShimmer(
    BuildContext context, {
    double width = 320,
    double ratioX = 4,
    double ratioY = 3,
  }) {
    final double height = width * ratioY / ratioX;
    return Shimmer(
      gradient: const LinearGradient(
        begin: Alignment.centerRight,
        end: Alignment.centerLeft,
        colors: [
          Color.fromRGBO(231, 235, 240, 1),
          Color.fromRGBO(231, 235, 240, 1),
          Color.fromRGBO(251, 251, 251, 0.8),
          Color.fromRGBO(231, 235, 240, 1),
          Color.fromRGBO(231, 235, 240, 1),
        ],
        stops: <double>[0, 0.35, 0.5, 0.65, 2],
      ),
      period: const Duration(seconds: 2),
      direction: ShimmerDirection.ltr,
      child: Container(
        width: width,
        height: height,
        decoration: BoxDecoration(
          borderRadius: BorderRadius.circular(16),
          color: Colors.grey,
        ),
      ),
    );
  }

  Widget _widgetWithImage(
    BuildContext context, {
    double width = 320,
    double ratioX = 4,
    double ratioY = 3,
    required String modelImg,
  }) {
    final double height = width * ratioY / ratioX;
    return ClipRRect(
      borderRadius: BorderRadius.circular(16),
      child: Stack(
        fit: StackFit.passthrough,
        children: [
          CachedNetworkImage(
            width: width,
            height: height,
            imageUrl: modelImg,
            fit: BoxFit.cover,
            placeholder: (_, __) => Image.asset(
              "assets/brand/placeholder.jpeg",
              width: width,
              height: height,
              fit: BoxFit.fitWidth,
            ),
            errorWidget: (_, __, ___) => Image.asset(
              "assets/png/covered_car.png",
              width: width,
              height: height,
              fit: BoxFit.fitHeight,
            ),
          ),
          GestureDetector(
            behavior: HitTestBehavior.opaque,
            onTap: onTapItem,
            child: Container(
              width: width,
              height: height,
              padding: EdgeInsets.all(14),
              color: Colors.black.withOpacity(0.3),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    crossAxisAlignment: CrossAxisAlignment.center,
                    children: [
                      Expanded(
                        child: Text(
                          '${carDetails!.brand} ${carDetails!.model}',
                          maxLines: 2,
                          overflow: TextOverflow.ellipsis,
                          style: AppStyle.textStyle().copyWith(
                            color: Colors.white,
                            fontSize: 20,
                            fontWeight: FontWeight.w600,
                          ),
                        ),
                      ),
                      SizedBox(width: 20),
                      Container(
                        padding: const EdgeInsets.symmetric(horizontal: 4),
                        decoration: BoxDecoration(
                          color: Color(0xffECECEC),
                          borderRadius: BorderRadius.circular(8),
                        ),
                        child: carDetails!.finesCount != 0
                            ? Text(
                                S.of(context).nFines(carDetails!.finesCount),
                                style: AppStyle.textStyle().copyWith(
                                  color: Color(0xFFFF0000),
                                  fontWeight: FontWeight.w600,
                                  fontSize: 20,
                                ),
                              )
                            : SizedBox(),
                      ),
                      SizedBox(width: 4),
                      _widgetPinned(),
                    ],
                  ),
                  Spacer(),
                  CarNumber(carNumber: carDetails!.number),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }

  Widget _widgetNoImage(
    BuildContext context, {
    double width = 320,
    double ratioX = 4,
    double ratioY = 3,
    String? brandImg,
    String? modelImg,
  }) {
    final double height = width * ratioY / ratioX;
    return Container(
      width: width,
      height: height,
      padding: EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(16.0),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            crossAxisAlignment: CrossAxisAlignment.center,
            children: [
              ClipRRect(
                borderRadius: BorderRadius.circular(12),
                child: CachedNetworkImage(
                  imageUrl: "$brandImg",
                  width: 40,
                  height: 40,
                  fit: BoxFit.fill,
                  errorWidget: (context, url, error) {
                    return Image.asset(
                      "assets/brand/placeholder.jpeg",
                      width: 40,
                      height: 40,
                      fit: BoxFit.fill,
                    );
                  },
                ),
              ),
              SizedBox(width: 4),
              Expanded(
                flex: 2,
                child: SizedBox(
                  height: 40,
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        carDetails!.brand,
                        style: AppStyle.textStyle().copyWith(
                            color: Color(0xff747474),
                            fontWeight: FontWeight.normal,
                            fontSize: 14.0),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                      Spacer(),
                      Text(
                        carDetails!.model,
                        style: AppStyle.textStyle().copyWith(
                            color: Color(0xff747474),
                            fontWeight: FontWeight.normal,
                            fontSize: 14.0),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ],
                  ),
                ),
              ),
              SizedBox(width: 4),
              carDetails!.finesCount != 0
                  ? Text(
                      S.of(context).nFines(carDetails!.finesCount),
                      style: AppStyle.textStyle().copyWith(
                        color: Color(0xFFFF0000),
                        fontWeight: FontWeight.w600,
                        fontSize: 20,
                      ),
                    )
                  : SizedBox(),
              SizedBox(width: 4),
              _widgetPinned(),
            ],
          ),
          Expanded(
            child: GestureDetector(
              behavior: HitTestBehavior.opaque,
              onTap: onTapItem,
              child: CachedNetworkImage(
                imageUrl: "$modelImg",
                width: double.infinity,
                fit: BoxFit.fitHeight,
                errorWidget: (context, url, error) {
                  return Image.asset(
                    "assets/png/covered_car.png",
                    fit: BoxFit.fitHeight,
                  );
                },
              ),
            ),
          ),
          GestureDetector(
            behavior: HitTestBehavior.opaque,
            onTap: onTapItem,
            child: Container(
              width: double.infinity,
              alignment: Alignment.center,
              child: CarNumber(carNumber: carDetails!.number),
            ),
          ),
        ],
      ),
    );
  }

  Widget _widgetPinned() {
    if (!isPinned) return SizedBox();
    return GestureDetector(
      behavior: HitTestBehavior.opaque,
      onTap: onTapPinned,
      child: Container(
        width: 40,
        height: 40,
        decoration: BoxDecoration(
          color: Color(0xffECECEC),
          shape: BoxShape.circle,
        ),
        child: Icon(
          carDetails!.pinned ? Icons.favorite : Icons.favorite_border,
          color: AppColors.unSelectedColor,
        ),
      ),
    );
  }
}
