import 'dart:async';

import 'package:bloc/bloc.dart';
import 'package:equatable/equatable.dart';
import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:road24_mobile/data/repository/CarsRepository.dart';
import 'package:road24_mobile/data/response/CarsResponse.dart';
import 'package:road24_mobile/model/models/AddCarModel.dart';
import 'package:road24_mobile/view/built/i18n.dart';
import 'package:road24_mobile/view/mycar/my_car_page.dart';

part 'cars_list_event.dart';

part 'cars_list_state.dart';

class CarsListBloc extends Bloc<CarsListEvent, CarsListState> {
  final CarsRepository _carsRepository = CarsRepository();
  final BuildContext context;

  CarsListBloc(this.context) : super(CarsListInitialState(false, []));

  final List<Results?> _fullList = [];

  @override
  Stream<CarsListState> mapEventToState(CarsListEvent event) async* {
    if (event is CarsChangedListEvent)
      yield* _mapChangedListEventToState(event);
    else if (event is CarsPinnedEvent)
      yield* _mapPinnedEventToState(event);
    else if (event is CarsRefreshEvent) yield* _mapRefreshEventToState(event);
  }

  Stream<CarsListState> _mapChangedListEventToState(
    CarsChangedListEvent event,
  ) async* {
    yield CarLoadingState();
    _fullList.clear();
    _fullList.addAll(event.fullList);
    yield CarRefreshSuccessState(_fullList);
  }

  Stream<CarsListState> _mapRefreshEventToState(CarsRefreshEvent event) async* {
    yield CarLoadingState();
    _fullList.clear();

    try {
      final json = await _carsRepository.getCars();
      final response = CarsResponse.fromJson(json);
      _fullList.addAll(response.results);
    } catch (e) {}
    print("response result...$_fullList");

    yield CarRefreshSuccessState(_fullList);
  }

  Stream<CarsListState> _mapPinnedEventToState(CarsPinnedEvent event) async* {
    MyCarPage.current = null;
    if (event.pinned &&
        _fullList.where((e) => e!.pinned).toList().length >= 5) {
      _log(S.of(event.context).add_car_not_info);
      return;
    }
    yield CarLoadingState();
    final carDetails = _fullList.where((e) => e!.id == event.carId).toList()[0];

    try {
      AddCarModel.carNumber = carDetails!.number;
      AddCarModel.tecNumber = carDetails.techPassNum;
      AddCarModel.tecSeries = carDetails.techPassSeries;
      AddCarModel.mark = carDetails.brand;
      AddCarModel.name = carDetails.model;
      AddCarModel.brandId = carDetails.carBrand?.id;
      AddCarModel.modelId = carDetails.carModel?.id;
      AddCarModel.imageId = carDetails.image?.id;

      await _carsRepository.editCar(event.carId, pinned: event.pinned);

      final json = await _carsRepository.getCars();

      final response = CarsResponse.fromJson(json);

      _fullList.clear();
      _fullList.addAll(response.results);
    } catch (e) {}
    yield CarRefreshSuccessState(_fullList);

    if (event.pinned)
      _log(S.of(context).add_car_info);
    else
      _log(S.of(context).remove_car_info);
  }

  void _log(String msg) {
    Fluttertoast.showToast(
      msg: msg,
      toastLength: Toast.LENGTH_LONG,
      timeInSecForIosWeb: msg.length ~/ 15,
    );
  }
}
