import 'dart:io';
import 'package:file_picker/file_picker.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter/rendering.dart';
import 'package:image_cropper/image_cropper.dart';
import 'package:image_picker/image_picker.dart';
import 'package:modal_bottom_sheet/modal_bottom_sheet.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:road24_mobile/model/utils/styles/AppColors.dart';
import 'package:road24_mobile/model/utils/styles/AppStyles.dart';
import 'package:road24_mobile/view/built/i18n.dart';
import 'package:road24_mobile/view/widgets/custom_dialog.dart';

mixin SelectPhotoImp {
  void onUpdateImage({String? fileName, File? file});
}

class SelectPhotoDialog extends StatelessWidget {
  final String title;
  final SelectPhotoImp imp;
  final double ratioX;
  final double ratioY;

  const SelectPhotoDialog._(this.title, this.imp, this.ratioX, this.ratioY);

  static void show(
    BuildContext context, {
    required String title,
    required SelectPhotoImp imp,
    double ratioX = 4,
    double ratioY = 3,
  }) {
    showBarModalBottomSheet(
      context: context,
      shape: RoundedRectangleBorder(
        borderRadius: BorderRadius.only(
          topLeft: Radius.circular(20),
          topRight: Radius.circular(20),
        ),
      ),
      builder: (_) => SelectPhotoDialog._(title, imp, ratioY, ratioY),
    );
  }

  void _setImage(BuildContext context, ImageSource source) async {
    var path;
    if (source == ImageSource.gallery) {
      var result = await FilePicker.platform.pickFiles(type: FileType.image);
      path = result?.files.single.path;
    } else {
      try {
        var result = await ImagePicker().pickImage(
          source: ImageSource.camera,
          imageQuality: 50,
        );
        path = result?.path;
      } catch (e) {
        CustomDialog(
          title: S.of(context).camera_denied,
          content: S.of(context).camera_denied_description,
          onYesTap: openAppSettings,
          onNoTap: () => Navigator.pop(context),
        ).showAlertDialog(context);
        return;
      }
    }
    if (path != null) {
      var file = File(path);
      if (file != null) {
        File? croppedFile = await ImageCropper.cropImage(
          sourcePath: file.path,
          aspectRatio: CropAspectRatio(ratioX: 3.5, ratioY: 3),
          compressQuality: 40,
        );

        print("image picked...$croppedFile");
        if (croppedFile != null) {
          imp.onUpdateImage(file: File(croppedFile.path), fileName: '$source');
        }
      }
    } else {
      // User canceled the picker
    }

    Navigator.pop(context);
  }

  @override
  Widget build(BuildContext context) {
    return Container(
      width: double.infinity,
      padding: EdgeInsets.symmetric(horizontal: 20),
      color: Colors.white,
      child: Column(
        mainAxisSize: MainAxisSize.min,
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          SizedBox(height: 20),
          Text(
            title,
            style: AppStyle.textStyle().copyWith(
              color: Colors.black,
              fontSize: 17,
              fontWeight: FontWeight.w600,
            ),
          ),
          SizedBox(height: 20),
          _widgetButton(
            text: S.of(context).open_camera,
            onTap: () => _setImage(context, ImageSource.camera),
          ),
          SizedBox(height: 10),
          _widgetButton(
            text: S.of(context).open_gallery,
            onTap: () => _setImage(context, ImageSource.gallery),
          ),
          SizedBox(height: 35),
        ],
      ),
    );
  }

  Widget _widgetButton({
    GestureTapCallback? onTap,
    String text = '',
  }) {
    return GestureDetector(
      onTap: onTap,
      child: Container(
        width: double.infinity,
        height: 50,
        padding: EdgeInsets.symmetric(horizontal: 30),
        decoration: BoxDecoration(
          color: Color(0xffE1E1E1),
          borderRadius: BorderRadius.circular(12),
        ),
        child: Row(
          children: [
            Expanded(
              child: Text(
                text,
                textAlign: TextAlign.center,
                style: AppStyle.textStyle().copyWith(
                  color: AppColors.selectedColor,
                  fontWeight: FontWeight.w600,
                  fontSize: 16,
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }
}
