import 'dart:ui';
import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import 'package:road24_mobile/data/repository/InspectionsRepository.dart';
import 'package:road24_mobile/data/response/manual_part/inspection/InspectionManualResponse.dart';
import 'package:road24_mobile/data/response/manual_part/inspection/RegionsModel.dart';
import 'package:road24_mobile/model/blocs/inspections/inspections_bloc.dart';
import 'package:road24_mobile/model/utils/styles/AppColors.dart';
import 'package:road24_mobile/model/utils/styles/AppStyles.dart';
import 'package:road24_mobile/view/built/i18n.dart';
import 'package:road24_mobile/view/manual/parts/inspection/InspectionPlaces/places_in_map_new.dart';
import 'package:road24_mobile/view/widgets/fade_route.dart';
import 'package:road24_mobile/view/widgets/no_connection_screen.dart';
import 'package:sizer/sizer.dart';

import '../../../../../my_app.dart';
import 'list_of_places_new.dart';

class InspectionPlaces extends StatefulWidget {
  final String regionName;
  final int regionId;

  const InspectionPlaces(
      {Key? key, required this.regionName, required this.regionId})
      : super(key: key);

  @override
  _InspectionPlacesState createState() => _InspectionPlacesState();
}

class _InspectionPlacesState extends State<InspectionPlaces>
    with SingleTickerProviderStateMixin {
  late TabController controller;
  int _activeTab = 0;
  List<Address> addressList = [];
  List<Districts> districtList = [];

  var inspectionBloc = InspectionsBloc(repository: InspectionsRepository());

  static const Map<int, LatLng> regionsLatLng = {
    1: LatLng(41.2825125, 69.1392828),
    2: LatLng(41.181598, 69.475480),
    3: LatLng(40.466194, 68.680332),
    4: LatLng(40.1189144, 67.7841833),
    5: LatLng(39.6407994, 66.8976754),
    6: LatLng(40.475911, 71.334335),
    7: LatLng(40.9705759, 71.5744109),
    8: LatLng(40.7792797, 72.1795614),
    9: LatLng(38.823695, 66.028248),
    10: LatLng(38.1281789, 66.3363041),
    11: LatLng(39.777597, 64.3876928),
    12: LatLng(40.0980699, 65.3365083),
    13: LatLng(41.2842881, 60.6482014),
    14: LatLng(42.4714806, 59.5336181),
  };
  // static const Map<int, LatLng> regionsLatLng2 = {
  //   1: LatLng(41.2825125, 69.1392828), //tosh sh
  //   2: LatLng(42.4714806, 59.5336181), //qora
  //   3: LatLng(40.7792797, 72.1795614), //and
  //   4: LatLng(39.777597, 64.3876928), //bux
  //   5: LatLng(40.1189144, 67.7841833), //jiz
  //   6: LatLng(38.7701532, 64.8829063), //qash
  //   7: LatLng(40.0980699, 65.3365083), //nav
  //   8: LatLng(40.9705759, 71.5744109), //nama
  //   9: LatLng(39.6407994, 66.8976754), //samar
  //   10: LatLng(38.1281789, 66.3363041), //surxon
  //   11: LatLng(40.8389774, 68.6352175), //sird
  //   12: LatLng(41.241491, 68.8339076), //tosh v
  //   13: LatLng(40.3389175, 70.6820173), //far
  //   14: LatLng(41.2842881, 60.6482014), //xor
  // };

  @override
  void initState() {
    controller = TabController(
      initialIndex: 0,
      length: 2,
      vsync: this,
    );

    controller
      ..addListener(() {
        setState(() {
          _activeTab = controller.index;
        });
      });

    inspectionBloc.add(GetListOfPlacesEvent(regionId: widget.regionId));
    super.initState();
  }

  @override
  void dispose() {
    controller.dispose();
    inspectionBloc.close();
    addressList.clear();
    super.dispose();
  }

  Widget _createTabForMain(String text, int index) {
    return Tab(
      child: Container(
        margin: const EdgeInsets.symmetric(horizontal: 5.0),
        padding: const EdgeInsets.symmetric(horizontal: 15.0),
        decoration: BoxDecoration(
          color: _activeTab == index ? Colors.white : Colors.transparent,
          borderRadius: BorderRadius.circular(100.0),
        ),
        child: Center(
          child: Text(
            text,
            style: TextStyle(
              color:
                  _activeTab == index ? Color(0xff000000) : Color(0xff747474),
              fontWeight:
                  _activeTab == index ? FontWeight.w600 : FontWeight.w500,
              fontSize: 15.0,
              height: 1.2,
              letterSpacing: 0.2,
            ),
            maxLines: 1,
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    // addressList.map((e) => print('location: ' + e.geolocatoin));
    return ValueListenableBuilder(
        valueListenable: MyApp.hasInternet,
        builder: (context, bool hasInternet, _) {
          return (hasInternet)
              ? GestureDetector(
                  onTap: () {
                    FocusScope.of(context).requestFocus(new FocusNode());
                  },
                  child: Scaffold(
                    backgroundColor: AppColors.appPrimaryColor,
                    appBar: appBar(context),
                    body: Container(
                      width: 100.0.w,
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.center,
                        children: [
                          const SizedBox(height: 20.0),
                          Container(
                            height: 50.0,
                            padding: const EdgeInsets.symmetric(vertical: 5.0),
                            margin:
                                const EdgeInsets.symmetric(horizontal: 16.0),
                            decoration: BoxDecoration(
                              color: Color(0xff767680).withOpacity(0.12),
                              borderRadius: BorderRadius.circular(100.0),
                            ),
                            child: TabBar(
                              isScrollable: false,
                              controller: controller,
                              indicatorColor: Colors.transparent,
                              labelPadding:
                                  const EdgeInsets.symmetric(horizontal: 0.0),
                              tabs: <Widget>[
                                _createTabForMain("${S.of(context).list}", 0),
                                _createTabForMain("${S.of(context).map}", 1),
                              ],
                            ),
                          ),
                          Expanded(
                            child: BlocBuilder(
                                bloc: inspectionBloc,
                                builder: (context, state) {
                                  if (state is LoadingState) {
                                    return const Center(
                                        child: CircularProgressIndicator());
                                  } else if (state is GetPlacesSuccessState) {
                                    addressList.clear();

                                    addressList.addAll(state.address);

                                    districtList.addAll(state.district);

                                    return TabBarView(
                                      physics:
                                          const NeverScrollableScrollPhysics(),
                                      controller: controller,
                                      children: <Widget>[
                                        ListOfPlaces(
                                          regionName: widget.regionName,
                                          listOfPlaces: addressList,
                                          districts: districtList,
                                        ),
                                        PlacesInMap(
                                          regionName: widget.regionName,
                                          listOfPlaces: addressList,
                                          defaultLocation:
                                              regionsLatLng[widget.regionId]!,
                                        ),
                                      ],
                                    );
                                  }
                                  return SizedBox();
                                }),
                          ),
                        ],
                      ),
                    ),
                  ),
                )
              : NoConnectionScreen(
                  onTap: () {
                    Navigator.pushReplacement(
                      context,
                      FadeRoute(
                          page: InspectionPlaces(
                        regionId: widget.regionId,
                        regionName: widget.regionName,
                      )),
                    );
                  },
                );
        });
  }

  AppBar appBar(BuildContext context) {
    return AppBar(
      toolbarHeight: 60.0,
      centerTitle: true,
      leading: Material(
        type: MaterialType.transparency,
        child: InkWell(
          onTap: () {
            Navigator.pop(context);
          },
          child: Container(
            margin: const EdgeInsets.all(10.0),
            decoration: const BoxDecoration(
              color: Color(0xffDFDFDF),
              shape: BoxShape.circle,
            ),
            child: const Center(
              child: Icon(
                Icons.arrow_back_ios_outlined,
                size: 20.0,
                color: Colors.black,
              ),
            ),
          ),
        ),
      ),
      title: Text(
        S.of(context).list_of_inspection_places,
        style: AppStyle.textStyle().copyWith(
            color: Colors.black, fontWeight: FontWeight.w600, fontSize: 21.0),
        maxLines: 2,
        textAlign: TextAlign.center,
      ),
    );
  }
}
