import os
from celery import Celery
from celery.schedules import crontab

# set the default Django settings module for the 'celery' program.
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'config.settings')

app = Celery('config')
redis_url = os.environ.get('REDIS_URL', 'redis://localhost:6379') + '/0'
app.conf.broker_url = redis_url

# Using a string here means the worker don't have to serialize
# the configuration object to child processes.
# - namespace='CELERY' means all celery-related configuration keys
#   should have a `CELERY_` prefix.
app.config_from_object('django.conf:settings', namespace='CELERY')

app.conf.beat_schedule = {
    'notify_discount_ending': {
        'task': 'apps.cars.tasks.task_notify_discount_ending',
        'schedule': crontab(minute=0, hour=10)
    },
    'notify_inspection_ending': {
        'task': 'apps.cars.tasks.notify_inspection_ending',
        'schedule': crontab(minute=0, hour=7)
    },
    'run-every-10-minutes': {
        'task': 'apps.cars.tasks.task_refresh_remain',
        'schedule': 10 * 60.0
    },
    'unsubscribe_cars': {
        'task': 'apps.cars.tasks.unsubscribe_cars',
        'schedule': 40 * 60.0  # run every 40 minutes
    },
    'get_failed_images': {
        'task': 'apps.cars.tasks.repeat_parse_pdf',
        'schedule': 40 * 60  # run every 40 minutes
    },
    'get_cars_date_of_registration': {
        'task': 'apps.cars.tasks.get_cars_date_of_registration',
        'schedule': 60 * 60  # run every 60 minutes
    },
    'telegram_bot_daily_reports': {
        'task': 'apps.bot.tasks.daily_reports',
        'schedule': crontab(minute=0, hour=0)  # Execute daily at midnight.
    }
}
app.conf.timezone = 'Asia/Tashkent'

# Load task modules from all registered Django app configs.
app.autodiscover_tasks()
