# Stdlib
import os

# Django
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework import status

# Project
from .models import Check
from apps.cars.models import Fine
from apps.bank_cards.models import BankCard
from apps.shared.utils import requests
from apps.notifications.services import (
    send_fine_notification,
    NotificationTemplate
)


def get_fine_check(pSeryNumber: str):
    url = 'https://core.unired.uz/api/v1/munis'
    headers = {"Authorization": "Bearer %s" % os.environ['MUNIS_TOKEN']}

    is_kv = pSeryNumber.upper().startswith('KV')
    supplier = "0210" if is_kv else "0206"
    element = "NDECREE"
    code = "03" if is_kv else "01"
    data = {
        "id": "1",
        "method": "munis.cheque.create",
        "params": {
            "settlement": {
                "supplier": supplier,
                "code": code
            },
            "subject": [
                {
                    "element": element,
                    "value": pSeryNumber
                }
            ]
        }
    }
    response = requests.post(url, json=data, headers=headers)
    return response


def pay_fine(bank_card: BankCard, amount: float, fine: Fine):
    from apps.cars.services.fines import refresh_remain

    fine, response = refresh_remain(fine)
    assert fine.pStatus == 1, _("You cannot pay this fine.")
    assert amount <= fine.remain, _("You can't pay more than remain.")
    assert response['status'], response['error']['message']
    tr_id = response['result']['tr_id']
    data = {
        "id": 1,
        "method": "munis.cheque.confirm",
        "params": {
            "tr_id": tr_id,
            "number": bank_card.number,
            "expire": bank_card.expire,
            "amount": str(int(amount * 100))
        }
    }

    url = 'https://core.unired.uz/api/v1/munis'
    headers = {"Authorization": "Bearer %s" % os.environ['MUNIS_TOKEN']}
    response = requests.post(url, json=data, headers=headers)

    assert response.status_code == status.HTTP_200_OK, 'Munis failed.'
    data = response.json()

    assert data['status'], data['error']['message']
    Check.objects.create(
        bank_card=bank_card,
        user=bank_card.user,
        fine=fine,
        state=data['result']['state'],
        result=data['result'],
        purpose_text=data['result']['info']['purpose']['text'],
        amount=data['result']['payer']['amount'],
        commission=data['result']['payer']['commission'],
        payee_details=data['result']['payee']
    )
    refresh_remain(fine)
    if fine.pStatus == 5:
        send_fine_notification(fine, NotificationTemplate.COMPLETELY_PAID_FINE)
