# Stdlib
import os
import io
from urllib.parse import urlencode
from random import randint
import json
import functools
from contextlib import contextmanager
import time
from hashlib import md5
from uuid import uuid4

# Django
from django.urls import reverse
from django.core.files.uploadedfile import SimpleUploadedFile
from django.conf import settings
from django.utils import timezone
from django.core.cache import cache

# Pypi: moviepy
from moviepy.editor import ImageClip

# Pypi: requests
import requests as original_requests
from requests.adapters import HTTPAdapter
from requests.exceptions import ReadTimeout

# Project
from apps.settings.models import Setting
from apps.logs.utils import (
    log_http_output_request, log_http_output_request_exception
)


def rev(url, params=None, **kwargs):
    """
    :param url:
    :param params: dict
    :return: full_url: full_url
    """
    return reverse(url, **kwargs) + '?' + urlencode(params or {})


def get_files_for_checking(start_path):
    source_files = []
    for root, dirs, files in os.walk(start_path, topdown=True):
        for file in files:
            if file.endswith('.py') and not root.endswith('migrations'):
                source_files.append('%s/%s' % (root, file))
    return source_files


def generate_image():
    file = io.BytesIO(
        b'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x01\x00\x00\x00\x01\x08\x02\x00\x00\x00\x90wS\xde\x00\x00\x00\x0cIDATx\x9cc```\x00\x00\x00\x04\x00\x01\xf6\x178U\x00\x00\x00\x00IEND\xaeB`\x82'  # noqa
    )
    file.name = 'test.png'
    file.seek(0)
    return file


def generate_video(duration=1):
    img = generate_image()
    with open('media/black_square.png', 'wb') as f:
        f.write(img.getvalue())

    clip = ImageClip('media/black_square.png')
    clip = clip.set_duration(duration)
    clip.write_videofile("media/video_test.mp4", fps=1)
    with open("media/video_test.mp4", 'rb') as file:
        return SimpleUploadedFile('hello.mp4', file.read())


def get_client_ip(request):
    x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
    if x_forwarded_for:
        ip = x_forwarded_for.split(',')[0]
    else:
        ip = request.META.get('REMOTE_ADDR')
    return ip


def _getsms_send_sms(phone, text):
    data = {
        'login': os.environ['GETSMS_LOGIN'],
        'password': os.environ['GETSMS_PASSWORD'],
        'data': json.dumps([{
            'phone': phone,
            'text': text
        }])
    }

    return requests.post('http://185.8.212.184/smsgateway/', json=data)


def _playmobile_send_sms(phone, text):
    hash_key = md5(str(uuid4()).encode()).hexdigest()[:17]
    data = {
        'messages': [
            {
                'recipient': phone,
                'message-id': 'anb%s' % hash_key,
                "sms": {
                    "originator": "3700",
                    "content": {"text": text}
                },
            }
        ]
    }

    return requests.post(
        'http://91.204.239.44/broker-api/send',
        json=data,
        auth=(
            os.environ['PLAYMOBILE_LOGIN'],
            os.environ['PLAYMOBILE_PASSWORD']
        )
    )


def send_sms(phone, text):
    if settings.ENV != 'production':
        return
    setting = Setting.objects.get()

    if setting.sms_gateway == Setting.GET_SMS:
        response = _getsms_send_sms(phone, text)
    else:
        response = _playmobile_send_sms(phone, text)
    return response


def send_sms_verification(instance, phone, text):
    sms_code = str(randint(100000, 999999))

    instance.sms_code = sms_code
    instance.sms_date = timezone.now()
    instance.save()

    send_sms(phone, text % sms_code)


def plural_days(n):
    days = ['день', 'дня', 'дней']

    if n % 10 == 1 and n % 100 != 11:
        p = 0
    elif 2 <= n % 10 <= 4 and (n % 100 < 10 or n % 100 >= 20):
        p = 1
    else:
        p = 2

    return days[p]


class MyAdapter(HTTPAdapter):
    def send(self, request, *args, **kwargs):
        try:
            return super().send(request, *args, **kwargs)
        except ReadTimeout as e:
            log_http_output_request_exception(request, 408, kwargs['timeout'])
            raise e
        except Exception as e:
            raise e


requests = original_requests.Session()
requests.mount('https://', MyAdapter())
requests.mount('http://', MyAdapter())
requests.request = functools.partial(requests.request, timeout=60, hooks={
    'response': log_http_output_request
})


@contextmanager
def memcache_lock(lock_id, oid):
    LOCK_EXPIRE = 60 * 10  # Lock expires in 10 minutes

    timeout_at = time.monotonic() + LOCK_EXPIRE - 3
    # cache.add fails if the key already exists
    status = cache.add(lock_id, oid, LOCK_EXPIRE)
    try:
        yield status
    finally:
        # memcache delete is very slow, but we have to use it to take
        # advantage of using add() for atomic locking
        if time.monotonic() < timeout_at and status:
            # don't release the lock if we exceeded the timeout
            # to lessen the chance of releasing an expired lock
            # owned by someone else
            # also don't release the lock if we didn't acquire it
            cache.delete(lock_id)
