# Django Rest Framework
from rest_framework import serializers

# Project
from apps.notifications.services import (
    NotificationTemplate, send_offence_notification
)
from ...models import Offence


class OffenceSerializer(serializers.Serializer):
    response = serializers.DictField(allow_null=True)
    id = serializers.CharField()
    extra_response = serializers.CharField(allow_null=True)
    status = serializers.ChoiceField(choices=Offence.STATUS_CHOICES)
    reward_status = serializers.ChoiceField(choices=Offence.REWARD_STATUS)

    def create(self, *args, **kwargs):
        data = self.initial_data
        response = data['response']
        pk = data['id']
        status = data['status']
        extra_response = data['extra_response']
        reward_status = data['reward_status']

        offence = Offence.objects.filter(ejarima_id=pk).first()

        if not offence:
            return data

        offence.extra_response = extra_response
        offence.reward_status = reward_status

        if status == Offence.REJECTED:
            langs = [
                {'lang': 'ru', 'prefix': 'ru'},
                {'lang': 'uz', 'prefix': 'uz_la'},
                {'lang': 'oz', 'prefix': 'uz_la'}
            ]
            for lang in langs:
                offence.set_current_language(lang['lang'])
                offence.reject_text = response['text_' + lang['prefix']]
                offence.reject_alias = response['alias_' + lang['prefix']]
                offence.save()

        offence.status = status
        offence.save()
        send_offence_notification(
            offence,
            NotificationTemplate.OFFENCE_CHANGED_STATUS
        )
        return offence
