# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Django
from django.urls import reverse

# Project
from ..models import Offence, Report
from apps.notifications.models import NotificationContent
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages


class ReportWebhooksTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'region.yaml',
        'district.yaml',
        'file.yaml',
        'address.yaml',
        'offences.yaml',
        'reports.yaml',
        'notification_template.yaml',
        'devices.yaml'
    ]

    report_url = reverse('webhook:report')
    offence_url = reverse('webhook:offence')

    def setUp(self):
        self._auth_ejarima()

    def test_unauth(self):
        self.client.credentials()

        response = self.client.post(self.report_url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self.client.post(self.offence_url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        self._auth_admin()
        response = self.client.post(self.report_url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self.client.post(self.offence_url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_format_report(self):
        data = {
            "id": "51a30504-b45d-47a2-8588-50c98dab45b6",
            "status": "lsdkgnlfgdgj"
        }
        response = self.client.post(self.report_url, data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "status": [
                messages.INVALID_CHOICE.format(input=data['status'])
            ]
        })

    def test_format_offence(self):
        data = {
            "status": "lsdkgnlfgdgj"
        }

        response = self.client.post(self.offence_url, data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "id": [messages.INVALID_REQUIRED],
            "extra_response": [messages.INVALID_REQUIRED],
            "reward_status": [messages.INVALID_REQUIRED],
            "response": [messages.INVALID_REQUIRED],
            "status": [
                messages.INVALID_CHOICE.format(input=data['status'])
            ]
        })

    def test_webhook_report(self):
        data = {
            "id": "51a30504-b45d-47a2-8588-50c98dab45b6",
            "status": Report.REVIEWED
        }
        response = self.client.post(self.report_url, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        report = Report.objects.get(id=1)
        self.assertEqual(report.status, data['status'])

    def test_webhook_offence_rejected(self):
        data = {
            'response': {
                'id': '980c94a7-120d-419d-8cb6-d9ca346399fa',
                'text_ru': 'В представленном видео не было обнаружено достат'
                           'очно признаков административного правонарушения',
                'text_uz_cy': 'Тақдим этилган видеоёзувда маъмурий хуқуқбуз'
                              'арлик ҳолати аниқланмади. Шу сабабли мурожаат'
                              ' кўриб чиқилмайди.',
                'number': 3,
                'text_uz_la': 'Taqdim etilgan videoyozuvda ma’muriy huquqb'
                              'uzarlik holati aniqlanmadi. Shu sababli m'
                              'urojaat ko’rib chiqilmaydi.\r\n',
                'alias_ru': 'нет правонарушений',
                'alias_uz_cy': 'хуқуқбузарлик йук',
                'alias_uz_la': "huqubuzarlik yo'q"
            },
            'vehicle_id_img': None,
            'accept_time': None, 'reward_status': 'absent',
            'fine_date': None,
            'citizen_id': '66b769c2-8964-4d6c-83d7-d3b9d3f098a4',
            'extra_img': None,
            'number': 252024, 'reward_amount': None, 'vehicle_id': '99H',
            'reject_time': '2021-02-12T16:39:42.948',
            'report_id': '1f340f7d-7001-4bcc-ac59-3ee508ec356a',
            'article': None,
            'testimony': 'Нарушение правил проезда остановок маршрутных тр'
                         'анспортных средств',
            'article_id': None, 'failure_time': None,
            'extra_response': None,
            'status': 'rejected',
            'id': '5a22a658-2850-4504-b1f0-38acf45ffaa7',
            'create_time': '2021-02-11T17:12:32.108', 'vehicle_img': None,
            'reward_params': {'no-reward': True}, 'fine': None,
            'pay_time': None,
            'dismiss_time': None, 'forward_time': None
        }
        response = self.client.post(self.offence_url, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        offence = Offence.objects.get(id=1)
        self.assertEqual(offence.status, data['status'])

        offence.set_current_language('ru')
        self.assertEqual(offence.reject_text, data['response']['text_ru'])
        self.assertEqual(offence.reject_alias, data['response']['alias_ru'])

        offence.set_current_language('uz')
        self.assertEqual(offence.reject_text, data['response']['text_uz_la'])
        self.assertEqual(offence.reject_alias, data['response']['alias_uz_la'])

    def test_webhook_offence_forwarded(self):
        data = {
            "response": None,
            "vehicle_id_img": "https://cdn.dyhxx.ejarima.uz/jarima/3891"
                              "018c-4269-447c-95d6-5ff1a89ee8c0.jpg?X-Amz-"
                              "Algorithm=AWS4-HMAC-SHA256&X-Amz-Credentia"
                              "l=jarima%2F20210220%2Fus-east-1%2Fs3%2Faws4"
                              "_request&X-Amz-Date=20210220T140031Z&X-Amz-Ex"
                              "pires=604800&X-Amz-SignedHeaders=host&X-Amz-"
                              "Signature=70d4cf258ef69b47d60d4e9c14df12f49d"
                              "f46aa4fa8b8216d0f74ef93a41ec1f",
            "accept_time": "2021-02-20T13:59:39.574",
            "reward_status": "absent",
            "fine_date": None,
            "citizen_id": "66b769c2-8964-4d6c-83d7-d3b9d3f098a4",
            "extra_img": None,
            "number": 264723,
            "reward_amount": None,
            "vehicle_id": "X",
            "reject_time": None,
            "report_id": "75b2243a-b915-4f70-8a48-c515bdff9c32",
            "article": {
                "alias_uz_cy": "тўхташ",
                "number": "128⁶-1",
                "text_uz_cy": "Транспорт воситалари ҳайдовчиларининг"
                              " тўхташ қоидаларини бузиши",
                "obsolete": False,
                "text_ru": "Нарушение водителями транспортных средст"
                           "в правил остановки",
                "text_uz_la": "Transport vositalari haydovchilarinin"
                              "g toʼxtash qoidalarini buzishi",
                "alias_uz_la": "toʼxtash",
                "factor": 3,
                "id": 45,
                "url": "http://lex.uz/docs/97661#1781987",
                "alias_ru": "остановка"
            },
            "testimony": "Нарушение правил перевозки людей",
            "article_id": 45,
            "failure_time": None,
            "extra_response": None,
            "status": "forwarded",
            "id": "5a22a658-2850-4504-b1f0-38acf45ffaa7",
            "create_time": "2021-02-20T13:52:30.189",
            "vehicle_img": "https://cdn.dyhxx.ejarima.uz/jarima/6c9d77e6-d66b-"
                           "4868-825b-1d411313f938.jpg?X-Amz-Algorithm=AWS"
                           "4-HMAC-SHA256&X-Amz-Credential=jarima%2F2021022"
                           "0%2Fus-east-1%2Fs3%2Faws4_request&X-Amz-Date=20210"
                           "220T140031Z&X-Amz-Expires=604800&X-Amz-SignedHe"
                           "aders=host&X-Amz-Signature=8f7a2e0bf1dcf2a1f"
                           "e1204698dd31edfb87a16ba9b6872598b0112e133f9524b",
            "reward_params": {
                "no-reward": True
            },
            "fine": None,
            "pay_time": None,
            "dismiss_time": None,
            "forward_time": "2021-02-20T13:59:40.501"
        }

        self.assertEqual(NotificationContent.objects.count(), 0)

        response = self.client.post(self.offence_url, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        offence = Offence.objects.get(id=1)
        self.assertEqual(offence.status, data['status'])

        self.assertEqual(NotificationContent.objects.count(), 1)

        notification = NotificationContent.objects.first()
        self.assertEqual(
            notification.content,
            {
                'body': 'ru_lang_desc',
                'data': {
                    'action': 'offence_changed_status',
                    'obj_id': 1,
                    'report_id': 1,
                    'click_action': 'FLUTTER_NOTIFICATION_CLICK'
                },
                'title': 'ru_lang_title'
            }
        )
