# Stdlib
from copy import deepcopy
from datetime import datetime, timedelta

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages
from ..models import Reward, Report


class ReportMobileTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'region.yaml',
        'district.yaml',
        'file.yaml',
        'address.yaml',
        'reports.yaml',
        'offences.yaml'
    ]

    url = 'mobile:report-%s'

    data = {
        'video': 5,
        'extra_video': 6,
        'incident_datetime': str(datetime.now()),
        'reward': {
            'kind': Reward.PHONE,
            'enter': '998999999999'
        },
        'address': {
            'latitude': 41.335459,
            'longitude': 69.376377,
        },
        'offences': [
            {
                'vehicle_id': 'AA12312',
                'testimony': 'asfb'
            },
            {
                'vehicle_id': 'AA12312',
                'testimony': 'asfb123'
            }
        ]
    }

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 1,
                    "address": {
                        "id": 1,
                        "latitude": 69.282998,
                        "longitude": 41.327307,
                        "landmark": "",
                        "district": {
                            "id": 1,
                            "name": "Учтепинский район",
                            "region": {
                                "id": 1,
                                "name": "Ташкент"
                            }
                        },
                        "address": "Amir Temur Avenue"
                    },
                    "status": Report.CREATED,
                    "video": {
                        "id": 1,
                        "name": "my_file",
                        "file": "http://testserver/media/test.jpg",
                        "video_preview": None,
                        "module": ""
                    },
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                }
            ]
        )

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "video": {
                    "id": 1,
                    "name": "my_file",
                    "file": "http://testserver/media/test.jpg",
                    "video_preview": None,
                    "module": ""
                },
                "status": "created",
                "extra_video": {
                    "id": 2,
                    "name": "my_file2",
                    "file": "http://testserver/media/test.jpg",
                    "video_preview": None,
                    "module": ""
                },
                "extra_video_type": "rear",
                "address": {
                    "id": 1,
                    "latitude": 69.282998,
                    "longitude": 41.327307,
                    "landmark": "",
                    "district": {
                        "id": 1,
                        "name": "Учтепинский район",
                        "region": {
                            "id": 1,
                            "name": "Ташкент"
                        }
                    },
                    "address": "Amir Temur Avenue"
                },
                "incident_datetime": "2017-01-01T00:00:00.100000+05:00",
                "offences": [
                    {
                        "id": 1,
                        "vehicle_id": "wi4r4uiryw74iu4",
                        "testimony": "prevysheniye skorosti",
                        "reward_status": "absent",
                        "status": "created",
                        "reject_text": None,
                        "reject_alias": None,
                        'ejarima_number': None
                    }
                ],
                "reward": None,
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                'ejarima_number': None
            }
        )

    def test_create_fail_required_fields(self):
        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                "video": [messages.INVALID_REQUIRED],
                "address": [messages.INVALID_REQUIRED],
                "incident_datetime": [messages.INVALID_REQUIRED],
                "offences": [messages.INVALID_REQUIRED],
                "reward": [messages.INVALID_REQUIRED],
            }
        )

    def test_create_fail_format_data(self):
        data = {
            'video': 1,
            'extra_video': 1,
            'extra_video_type': 123,
            'incident_datetime': 123,
            'reward': 1,
            'address': {},
            'offences': []
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)

        self.assertEqual(
            response.data,
            {
                "video": ["Wrong file"],
                "extra_video": ["Wrong file"],
                "extra_video_type": [
                    messages.INVALID_CHOICE.format(
                        input=data['extra_video_type']
                    )
                ],
                "address": {
                    "latitude": [messages.INVALID_REQUIRED],
                    "longitude": [messages.INVALID_REQUIRED],
                },
                "incident_datetime": [messages.incorrect_datetime_format()],
                "offences": ["Min length 1."],
                "reward": {
                    "non_field_errors": [
                        messages.INVALID_DICT.format(
                            datatype='int'
                        )
                    ]
                }
            }
        )

    def test_create_fail_incident_datetime_more_than_now(self):
        data = deepcopy(self.data)
        data['incident_datetime'] = str(datetime.now() + timedelta(minutes=1))
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'incident_datetime': [
                'Must be less than current time or not older than 3 days.'
            ]
        })

    def test_create_fail_incident_datetime_3_days_ago(self):
        data = deepcopy(self.data)
        date = datetime.now().replace(hour=20, minute=32) - timedelta(days=3)
        data['incident_datetime'] = str(date)
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'incident_datetime': [
                'Must be less than current time or not older than 3 days.'
            ]
        })

    def test_create_fail_incident_datetime_less_3_days(self):
        data = deepcopy(self.data)
        data['incident_datetime'] = str(datetime.now() - timedelta(days=3))
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'incident_datetime': [
                'Must be less than current time or not older than 3 days.'
            ]
        })

    def test_create_success(self):
        data = self.data
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        report = Report.objects.get(id=response.data['id'])
        self.assertEqual(report.status, Report.REVIEWED)
        self.assertEqual(report.extra_video_type, Report.SEQUEL)
        self.assertEqual(report.reward.kind, data['reward']['kind'])
        self.assertEqual(report.reward.enter, data['reward']['enter'])

        offences = report.offences.order_by('id')
        self.assertEqual(len(offences), 2)
        self.assertEqual(
            offences[0].testimony,
            data['offences'][0]['testimony']
        )
        self.assertEqual(
            offences[1].testimony,
            data['offences'][1]['testimony']
        )

    def test_create_fail_user_first_name_last_name(self):
        self._auth_non_admin()
        data = self.data
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'non_field_errors': ['Please add your first name and last name.']}
        )

    def test_create_fail_reward_enter_required(self):
        data = deepcopy(self.data)
        del data['reward']['enter']

        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {
                    'enter': [messages.INVALID_REQUIRED]
                }
            }
        )

    def test_create_success_with_no_reward(self):
        data = deepcopy(self.data)
        data['reward']['kind'] = Reward.NO_REWARD
        del data['reward']['enter']

        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        report = Report.objects.get(id=response.data['id'])
        self.assertEqual(report.reward.kind, data['reward']['kind'])
        self.assertEqual(report.reward.enter, "true")

    def test_fix_bug_create_extra_video_none(self):
        data = deepcopy(self.data)
        data['extra_video'] = None

        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

    def test_create_fail_offence_format_min_length(self):
        data = deepcopy(self.data)
        data['offences'][0]['vehicle_id'] = '1'
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)

        self.assertEqual(
            response.data,
            {
                'offences': [
                    {
                        'vehicle_id': [
                            messages.INVALID_MIN_LENGTH.format(min_length=5)
                        ]
                    },
                    {}
                ]
            }
        )

    def test_create_fail_offence_format_only_letters_and_numbers(self):
        data = deepcopy(self.data)
        data['offences'][0]['vehicle_id'] = '%%%%%'
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'offences': [
                    {'vehicle_id': ['Only letters and numbers.']},
                    {}
                ]
            }
        )

    def test_create_fail_card_reward_enter_max_min_length(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.CARD,
            'enter': '777'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Min and max length must be 16.']},
            }
        )

    def test_create_fail_card_reward_enter_only_letters(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.CARD,
            'enter': '111111111111111$'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Must contain only letters.']},
            }
        )

    def test_create_fail_card_reward_enter_start_with_8600_9860(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.CARD,
            'enter': '1111111111111111'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Must start from 8600 or 9860.']},
            }
        )

    def test_create_success_card_reward_enter_start_with_8600_9860(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.CARD,
            'enter': '8600111111111111'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        data['reward'] = {
            'kind': Reward.CARD,
            'enter': '9860111111111111'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

    def test_create_fail_phone_only_numbers(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.PHONE,
            'enter': '+998 (99) 999-99-99'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Must contain only numbers.']},
            }
        )

    def test_create_fail_phone_min_max_len_12(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.PHONE,
            'enter': '999999999999999'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Min and max length must be 12.']},
            }
        )

    def test_create_fail_phone_start_998(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.PHONE,
            'enter': '333333333333'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Must start from 998.']},
            }
        )

    def test_create_fail_fund(self):
        data = deepcopy(self.data)
        data['reward'] = {
            'kind': Reward.FUND,
            'enter': '333333333333'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'reward': {'enter': ['Wrong fund name.']},
            }
        )
