# Django
from django.db import transaction
from django.utils.timezone import localtime

# Pypi: celery
from celery import shared_task

# Project
from apps.reports.utils.ejarima import Ejarima
from apps.reports.models import Report, Offence


@shared_task
@transaction.atomic
def send_to_ejarima(report_pk):
    ejarima = Ejarima()
    report = Report.objects.get(id=report_pk)
    offences = report.offences.order_by('id')
    address = report.address
    reward = report.reward
    incident_time = localtime(report.incident_datetime).strftime('%H:%M')
    incident_date = localtime(report.incident_datetime).strftime('%d.%m.%Y')

    data = {
        "offenses": {
            str(key): {
                'vehicle_id': value.vehicle_id,
                'testimony': value.testimony
            } for key, value in enumerate(offences)
        },
        "lat": address.latitude,
        "lng": address.longitude,
        "address": address.address,
        "incident_date": incident_date,
        "incident_time": incident_time,
        "area_id": address.district.region.ejarima_id,
        "district_id": address.district.ejarima_id,
        "with_extra_video": False,
        "reward_params": {
            reward.kind: reward.enter
        }
    }

    file = report.video.file
    file = ejarima.upload_video(file.name, open(file.path, 'rb'))
    data['video_id'] = file['id']

    if extra_video := report.extra_video:
        file = extra_video.file
        file = ejarima.upload_video(file.name, open(file.path, 'rb'))
        data['extra_video_id'] = file['id']
        data['with_extra_video'] = True
        data['extra_video_type'] = report.extra_video_type

    ejarima_report = ejarima.create_report(data, report.created_by)
    report.ejarima_id = ejarima_report['id']
    report.ejarima_number = ejarima_report['number']
    report.save()

    ejarima_offences = ejarima.get_offences(ejarima_report['id'])['results']
    for key, value in enumerate(ejarima_offences):
        offences[key].ejarima_id = value['id']
        offences[key].ejarima_number = value['number']
    Offence.objects.bulk_update(offences, ['ejarima_id', 'ejarima_number'])
