# Stdlib
from datetime import timedelta

# Django
from django.utils import timezone
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.geo.serializers import AddressSerializer
from apps.files.serializers import FileSerializer, File
from ...models import Report
from ...services import create_report
from .offence import RewardSerializer, OffenceSerializer


class ReportListSerializer(serializers.ModelSerializer):
    address = AddressSerializer()
    video = FileSerializer()

    class Meta:
        model = Report
        fields = (
            'id',
            'status',
            'address',
            'video',
            'created_date',
        )


class ReportSerializer(serializers.ModelSerializer):
    address = AddressSerializer()
    reward = RewardSerializer()
    offences = OffenceSerializer(many=True)

    def to_representation(self, instance):
        self.fields['video'] = FileSerializer()
        self.fields['extra_video'] = FileSerializer()
        return super().to_representation(instance)

    class Meta:
        model = Report
        fields = (
            'id',
            'video',
            'status',
            'extra_video',
            'extra_video_type',
            'address',
            'incident_datetime',
            'offences',
            'reward',
            'created_date',
            'ejarima_number'
        )
        read_only_fields = ('status', 'ejarima_number')

    def validate_offences(self, offences):
        if len(offences) < 1:
            raise serializers.ValidationError('Min length 1.')

        return offences

    def validate_video(self, video):
        if video.module != File.REPORT:
            raise serializers.ValidationError('Wrong file')
        return video

    def validate_extra_video(self, extra_video):
        if extra_video and extra_video.module != File.REPORT:
            raise serializers.ValidationError('Wrong file')
        return extra_video

    def validate_incident_datetime(self, incident_datetime):
        now = timezone.now()
        min_date = (now - timedelta(days=2)).date()

        is_more_then_now = incident_datetime > now

        if not (incident_datetime.date() > min_date and not is_more_then_now):
            msg = _('Must be less than current time or not older than 3 days.')
            raise serializers.ValidationError(msg)
        return incident_datetime

    def validate(self, attrs):
        user = self.context['request'].user
        if not user.phone:
            raise serializers.ValidationError(_('You must have phone.'))

        if not user.last_name or not user.first_name:
            msg = _('Please add your first name and last name.')
            raise serializers.ValidationError(msg)
        return attrs

    def update(self, instance, validated_data):
        raise NotImplemented

    def create(self, validated_data):
        validated_data['created_by'] = self.context['request'].user
        try:
            return create_report(**validated_data)
        except AssertionError as e:
            raise serializers.ValidationError({
                'non_field_errors': [e]
            })
