# Django
from django.utils.translation import gettext as _
from django.conf import settings

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.shared.validators import OnlyLettersAndNumbersValidator
from apps.bank_cards.utils import check_bank_card
from ...models import Reward, Offence, Fund


class FundSerializer(serializers.ModelSerializer):
    class Meta:
        model = Fund
        fields = ('id', 'name', 'link')


class RewardSerializer(serializers.ModelSerializer):
    class Meta:
        model = Reward
        fields = ('id', 'kind', 'enter')

    def validate(self, attrs):
        request = self.context['request']
        language = request.LANGUAGE_CODE
        kind = attrs['kind']
        enter = attrs.get('enter')

        if kind == Reward.NO_REWARD:
            attrs['enter'] = "true"
        elif not enter:
            raise serializers.ValidationError({
                'enter': [_("This field is required.")]
            })

        if kind == Reward.FUND:
            if enter not in Fund.objects.values_list('name', flat=True):
                raise serializers.ValidationError({
                    'enter': [_("Wrong fund name.")]
                })

        if kind == Reward.PHONE:
            if not all(i.isdigit() for i in enter):
                raise serializers.ValidationError({
                    'enter': [_("Must contain only numbers.")]
                })

            if len(enter) != 12:
                raise serializers.ValidationError({
                    'enter': [_("Min and max length must be 12.")]
                })

            if not enter.startswith('998'):
                raise serializers.ValidationError({
                    'enter': [_("Must start from 998.")]
                })

        if kind == Reward.CARD:
            if len(enter) != 16:
                raise serializers.ValidationError({
                    'enter': [_("Min and max length must be 16.")]
                })
            if not all(i.isdigit() for i in enter):
                raise serializers.ValidationError({
                    'enter': [_("Must contain only letters.")]
                })

            if not enter.startswith('8600') and not enter.startswith('9860'):
                raise serializers.ValidationError({
                    'enter': [_("Must start from 8600 or 9860.")]
                })

            if settings.ENV == 'production':
                is_success, reason = check_bank_card(enter)
                if not is_success:
                    lang_error = 'ru' if language == 'ru' else 'uz'

                    raise serializers.ValidationError({
                        'enter': [reason[lang_error]]
                    })

        return attrs


class OffenceSerializer(serializers.ModelSerializer):
    class Meta:
        model = Offence
        fields = (
            'id',
            'vehicle_id',
            'testimony',
            'reward_status',
            'status',
            'reject_text',
            'reject_alias',
            'ejarima_number'
        )
        read_only_fields = ('ejarima_number',)
        extra_kwargs = {
            'vehicle_id': {
                'min_length': 5,
                'validators': [OnlyLettersAndNumbersValidator()]
            }
        }
