# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.problems.models import Problem, Vote
from apps.geo.models import Address
from apps.shared.tests import ViewSetTestCase, AuthTestCase


class ProblemMobileTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'region.yaml',
        'district.yaml',
        'file.yaml',
        'address.yaml',
        'problem.yaml',
        'vote.yaml'
    ]

    url = 'mobile:problem-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 11)
        self.assertEqual(len(response.data['results']), 10)
        self.assertEqual(
            response.data['results'][0],
            {
                "id": 11,
                "description": "Hello",
                "kind": "other",
                "address": {
                    "district": {
                        "id": 1,
                        "region": {
                            "id": 1,
                            "name": "Ташкент",
                        },
                        "name": "Учтепинский район"
                    },
                    "address": "Temur Malik Street1"
                },
                'status': Problem.MODERATING,
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                "votes_count": 0
            }
        )

    def test_list_unauth(self):
        self.client.credentials()
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 11)
        self.assertEqual(len(response.data['results']), 2)

    def test_filter_kind(self):
        response = self._list({'kind': Problem.MARKUP})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 1)
        self.assertEqual(len(response.data['results']), 1)
        self.assertEqual(response.data['results'][0]['id'], 2)

    def test_filter_distance(self):
        response = self._list({'distance': '41.327590,69.283390,1000'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)

        ids = [i['id'] for i in response.data['results']]
        self.assertEqual(ids, [2, 1])

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "description": "Hello",
                "kind": "other",
                "photos": [
                    {
                        "id": 2,
                        "name": "my_file2",
                        "file": "http://testserver/media/test.jpg",
                        "video_preview": None,
                        "module": ""
                    },
                    {
                        "id": 1,
                        "name": "my_file",
                        "file": "http://testserver/media/test.jpg",
                        "video_preview": None,
                        "module": ""
                    }
                ],
                "address": {
                    "id": 1,
                    "latitude": 69.282998,
                    "longitude": 41.327307,
                    "landmark": "",
                    "district": {
                        "id": 1,
                        "name": "Учтепинский район",
                        "region": {
                            "id": 1,
                            "name": "Ташкент"
                        }
                    },
                    "address": "Amir Temur Avenue"
                },
                "status": "accepted",
                "comment": "we promise we will solve this issue",
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                "votes_count": 0,
                "is_vote": False,
                "hash_key": None
            }

        )

    def test_retrieve_voted(self):
        response = self._retrieve({'pk': 9})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['is_vote'], True)

    def test_retrieve_voted_not_auth(self):
        self.client.credentials()
        response = self._retrieve({'pk': 9})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['is_vote'], False)

    def test_create(self):
        self.assertEqual(Problem.objects.count(), 12)
        self.assertEqual(Address.objects.count(), 12)

        data = {
            'address': {
                'longitude': 69.376377,
                'latitude': 41.335459,
                'landmark': 'My home'
            },
            'kind': Problem.TRAFFIC_LIGHT,
            'description': "No Traffic light",
            'photos': [1, 2]
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Problem.objects.count(), 13)
        self.assertEqual(Address.objects.count(), 13)

        problem = Problem.objects.get(id=response.data['id'])
        self.assertEqual(problem.kind, data['kind'])
        self.assertEqual(problem.description, data['description'])
        self.assertEqual(problem.photos.count(), len(data['photos']))
        self.assertIsNotNone(problem.hash_key)

        address = problem.address
        self.assertEqual(address.district_id, 2)
        self.assertEqual(address.longitude, data['address']['longitude'])
        self.assertEqual(address.latitude, data['address']['latitude'])
        self.assertEqual(address.landmark, data['address']['landmark'])
        self.assertEqual(
            address.address,
            'Ташкент, Мирзо-Улугбекский район, Карасу 1 массив, 34'
        )

    # def test_vote_fail_already(self):
    #     response = self._create({}, {'method': 'vote', 'pk': 9})
    #     self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
    #     self.assertEqual(
    #         response.data, {"non_field_errors": ['Already voted.']}
    #     )
    #
    # def test_vote(self):
    #     self.assertEqual(Vote.objects.count(), 2)
    #     response = self._create({}, {'method': 'vote', 'pk': 1})
    #     self.assertEqual(response.status_code, status.HTTP_200_OK)
    #     self.assertEqual(
    #         response.data,
    #         {'downvotes_count': 0, 'votes_for_count': 1}
    #     )
    #     self.assertEqual(Vote.objects.count(), 3)
    #
    # def test_downvote_fail_already(self):
    #     response = self._create({}, {'method': 'downvote', 'pk': 10})
    #     self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
    #     self.assertEqual(
    #         response.data, {"non_field_errors": ['Already downvoted.']}
    #     )
    #
    #
    # def test_vote_change_to_downvote(self):
    #     self.assertEqual(Vote.objects.count(), 2)
    #     vote = Vote.objects.get(id=1)
    #     self.assertEqual(vote.is_vote, True)
    #
    #     response = self._create({}, {'method': 'downvote', 'pk': 9})
    #     self.assertEqual(response.status_code, status.HTTP_200_OK)
    #     self.assertEqual(Vote.objects.count(), 2)
    #     vote.refresh_from_db()
    #     self.assertEqual(vote.is_vote, False)
