# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.problems.models import Problem
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages


class ProblemAdminTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'region.yaml',
        'district.yaml',
        'file.yaml',
        'address.yaml',
        'problem.yaml',
        'vote.yaml'
    ]

    url = 'myadmin:problem-%s'

    def setUp(self):
        self._auth_admin()

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 12)
        self.assertEqual(len(response.data['results']), 10)
        self.assertEqual(
            response.data['results'][0],
            {
                "id": 12,
                "description": "Hello",
                "kind": "other",
                "address": {
                    "address": "Temur Malik Street1",
                    "district": {
                        "id": 1,
                        "region": {
                            "id": 1,
                            "name": "Ташкент",
                        },
                        "name": "Учтепинский район"
                    }
                },
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                "votes_count": 0
            }
        )

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 12)
        self.assertEqual(len(response.data['results']), 2)

    def test_filter_kind(self):
        response = self._list({'kind': Problem.MARKUP})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 1)
        self.assertEqual(len(response.data['results']), 1)
        self.assertEqual(response.data['results'][0]['id'], 2)

    def test_filter_distance(self):
        response = self._list({'distance': '41.327590,69.283390,1000'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)

        ids = [i['id'] for i in response.data['results']]
        self.assertEqual(ids, [2, 1])

    def test_update(self):
        data = {
            'status': Problem.REJECTED,
            'comment': "We don't know why"
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        problem = Problem.objects.get(id=response.data['id'])
        self.assertEqual(problem.status, data['status'])
        self.assertEqual(problem.comment, data['comment'])

    def test_update_require_comment(self):
        data = {
            'status': Problem.ACCEPTED
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'comment': [messages.INVALID_REQUIRED]
        })
