# Django
from django.db.models import Q
from django.utils.decorators import method_decorator
from django.views.decorators.cache import cache_page

# Django Rest Framework
from rest_framework import mixins
from rest_framework.validators import ValidationError
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.viewsets import GenericViewSet
from rest_framework.permissions import IsAuthenticatedOrReadOnly

# Project
from .serializers import ProblemSerializer, ProblemListSerializer
from ..services import vote_problem
from ..models import Problem
from ..filter import ProblemFilter


class ProblemViewSet(mixins.ListModelMixin,
                     mixins.RetrieveModelMixin,
                     mixins.CreateModelMixin,
                     GenericViewSet):
    permission_classes = (IsAuthenticatedOrReadOnly,)
    serializer_class = ProblemSerializer
    filterset_class = ProblemFilter

    def get_queryset(self):
        user = self.request.user
        queryset = Problem.objects.select_related(
            'address__district__region'
        ).order_by('-id')

        statuses = [Problem.MODERATING, Problem.REJECTED]

        if user.is_anonymous:
            queryset = queryset.exclude(status__in=statuses)
        else:
            queryset = queryset.exclude(
                Q(status__in=statuses) & ~Q(created_by=self.request.user)
            )

        if self.action == 'retrieve':
            queryset = queryset.prefetch_related('photos')
        return queryset

    def get_serializer_class(self):
        if self.action == 'list':
            return ProblemListSerializer
        return ProblemSerializer

    @method_decorator(cache_page(60 * 60 * 2))
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @action(methods=['POST'], detail=True)
    def vote(self, request, *args, **kwargs):
        instance = self.get_object()
        try:
            return Response(vote_problem(instance, request.user))
        except AssertionError as e:
            raise ValidationError({'non_field_errors': [e]})
