# Stdlib
from hashlib import md5
from uuid import uuid4

# Django
from django.db import transaction

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.geo.serializers import AddressSerializer
from apps.geo.services import address_create
from apps.files.serializers import FileSerializer
from ..models import Problem


class ProblemSerializer(serializers.ModelSerializer):
    address = AddressSerializer(
        help_text="""
        {longitude: 41.306581, latitude: 69.339555,landmark: 'My home'}
        """
    )
    is_vote = serializers.SerializerMethodField()

    def get_is_vote(self, obj):
        user = self.context['request'].user
        if user.is_anonymous:
            return False
        return obj.votes.filter(user=user).exists()

    def to_representation(self, instance):
        self.fields['photos'] = FileSerializer(many=True)
        return super().to_representation(instance)

    class Meta:
        model = Problem
        fields = (
            'id',
            'description',
            'kind',
            'photos',
            'address',
            'status',
            'comment',
            'created_date',
            'votes_count',
            'is_vote',
            'hash_key'
        )
        extra_kwargs = {
            'hash_key': {'read_only': True},
            'votes_count': {'read_only': True},
        }

    def update(self, instance, validated_data):
        raise NotImplemented

    @transaction.atomic
    def create(self, validated_data):
        user = self.context['request'].user
        address = validated_data['address']

        try:
            validated_data['address'] = address_create(**address)
        except AssertionError as e:
            raise serializers.ValidationError({'address': [e]})

        hash_key = md5(str(uuid4()).encode()).hexdigest()[:10]
        validated_data['hash_key'] = hash_key
        validated_data['created_by'] = user
        return super().create(validated_data)


class ProblemListSerializer(serializers.ModelSerializer):
    address = AddressSerializer(fields=('address', 'district',))

    class Meta:
        model = Problem
        fields = (
            'id',
            'description',
            'kind',
            'address',
            'status',
            'created_date',
            'votes_count'
        )
