# Django
from django.utils.translation import ugettext as _
from django.db import transaction

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.geo.serializers import AddressSerializer
from apps.files.serializers import FileSerializer
from ..models import Problem


class ProblemSerializer(serializers.ModelSerializer):
    address = AddressSerializer(
        help_text="""
        {longitude: 41.306581, latitude: 69.339555,landmark: 'My home'}
        """
    )

    def to_representation(self, instance):
        self.fields['photos'] = FileSerializer(many=True)
        return super().to_representation(instance)

    class Meta:
        model = Problem
        ref_name = 'admin'
        fields = (
            'id',
            'description',
            'status',
            'comment',
            'kind',
            'address',
            'created_date',
            'votes_count',
            'hash_key',
            'photos',
        )
        extra_kwargs = {
            'hash_key': {'read_only': True}
        }

    def validate(self, attrs):
        comment = attrs.get('comment')
        status = attrs.get('status')
        if not comment and status in [Problem.ACCEPTED, Problem.REJECTED]:
            raise serializers.ValidationError({
                'comment': [_('This field is required.')]}
            )
        return attrs


class ProblemListSerializer(serializers.ModelSerializer):
    address = AddressSerializer(fields=('address', 'region',))

    class Meta:
        model = Problem
        ref_name = 'admin'
        fields = (
            'id',
            'description',
            'kind',
            'address',
            'created_date',
            'votes_count'
        )
