# Django Rest Framework
from rest_framework.generics import CreateAPIView
from rest_framework.response import Response
from rest_framework import serializers

# Project
from apps.cars.models import Fine
from apps.reports.models import Report, Offence
from apps.notifications.services import (
    NotificationTemplate,
    send_offence_notification,
    send_fine_notification,
    send_report_notification
)


class SerializerAction(serializers.Serializer):
    action = serializers.ChoiceField(
        choices=NotificationTemplate.ACTION_CHOICES
    )
    fine = serializers.PrimaryKeyRelatedField(
        queryset=Fine.objects.all(),
        allow_null=True
    )
    report = serializers.PrimaryKeyRelatedField(
        queryset=Report.objects.all(),
        allow_null=True
    )
    offence = serializers.PrimaryKeyRelatedField(
        queryset=Offence.objects.all(),
        allow_null=True
    )

    def create(self, validated_data):
        fine = validated_data.get('fine')
        report = validated_data.get('report')
        offence = validated_data.get('offence')
        action = validated_data['action']

        if fine:
            send_fine_notification(fine, action)

        if report:
            send_report_notification(report, action)

        if offence and action == NotificationTemplate.OFFENCE_CHANGED_STATUS:
            send_offence_notification(offence, action)

        return validated_data


class NotificationEmulatorView(CreateAPIView):
    serializer_class = SerializerAction
