# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from ..models import Notification


class NotificationTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'notification_template.yaml',
        'notification_content.yaml',
        'notifications.yaml'

    ]

    url = 'mobile:notifications-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list(kwargs={'method': 'count'})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._update({'pk': 1}, {})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 4)
        self.assertEqual(len(response.data['results']), 4)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 4,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 3,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 2,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 1,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                }
            ]
        )

    def test_list_lang_uz(self):
        response = self._list(kwargs={'headers': {
            'HTTP_ACCEPT_LANGUAGE': 'uz'
        }})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 4)
        self.assertEqual(len(response.data['results']), 4)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 4,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 3,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 2,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 1,
                    "notification_content": {
                        "id": 1,
                        "action": "partial_paid_fine",
                        "content": {
                            "hello": "RU"
                        }
                    },
                    "is_read": False,
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                }
            ]
        )

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "notification_content": {
                    "id": 1,
                    "action": "partial_paid_fine",
                    "content": {
                        "hello": "RU"
                    }
                },
                "is_read": False,
                "created_date": "2017-01-01T00:00:00.100000+05:00"
            }
        )

    def test_update(self):
        notification = Notification.objects.get(id=1)
        self.assertEqual(notification.notification_content_id, 1)
        self.assertFalse(notification.is_read)

        data = {
            'notification_content': 2,
            'is_read': True
        }
        response = self._update({'pk': 1}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        notification.refresh_from_db()
        self.assertEqual(notification.notification_content_id, 1)
        self.assertTrue(notification.is_read)

    def test_create(self):
        response = self._create({})
        self.assertEqual(
            response.status_code,
            status.HTTP_405_METHOD_NOT_ALLOWED
        )

    def test_destroy(self):
        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertEqual(Notification.objects.filter(id=1).exists(), False)

    def test_count(self):
        response = self._list(kwargs={'method': 'count'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {'total_count': 4, 'unread_count': 4})
