# Stdlib
import itertools
from string import Template

# Django
from django.conf import settings
from django.db import transaction

# Pypi: django-parler
from django.db.models import QuerySet
from parler.utils.context import switch_language

# Pypi: fcm_django
from fcm_django.models import FCMDevice

# Project
from apps.cars.models import Fine, Inspection, Confidant, Insurance, Tinting
from apps.reports.models import Report, Offence
from .models import Notification, NotificationTemplate, NotificationContent


def _send_data_notifications(notifications: list[Notification]):
    if settings.ENV != 'production':
        return

    for notification in notifications:
        user = notification.user
        devices = FCMDevice.objects.filter(user=user)
        for device in devices:
            language = device.language or user.language
            notification_content = notification.notification_content

            with switch_language(notification_content, language):
                data = notification_content.content
            data['data']['notification_content_pk'] = notification_content.id

            device.send_message(**data)


def _create_content(obj, users, action, template_data=None, data=None):
    if not template_data:
        template_data = {}

    if not data:
        data = {'obj_id': obj.id}

    data['click_action'] = "FLUTTER_NOTIFICATION_CLICK"
    data['action'] = action

    template = NotificationTemplate.objects.get(action=action)
    languages = settings.LANGUAGES
    content = NotificationContent.objects.create(
        content_object=obj,
        template=template
    )

    for language in languages:
        with switch_language(template, language[0]):
            title = template.title
            body = template.body

        content.set_current_language(language[0])
        content.content = {
            'title': Template(title).substitute(template_data),
            'body': Template(body).substitute(template_data),
            'data': data
        }
        content.save()

    return Notification.objects.bulk_create(
        Notification(
            user=user,
            notification_content=content
        ) for user in users
    )


@transaction.atomic
def send_fine_notification(fine: Fine, action, extra_template_data=None):
    template_data = {
        'pAmount': fine.pAmount,
        'pStatus': fine.pStatus,
        'remain': fine.remain,
        'pDate': fine.pDate,
        'pSeryNumber': fine.pSeryNumber,
        'created_date': str(fine.created_date.strftime('%d.%m.%Y %H:%M:%S'))
    }
    if extra_template_data:
        template_data.update(extra_template_data)

    if fine.pStatus == 4 and fine.parent:
        template_data['originalAmount'] = fine.parent.pAmount
        template_data['remain'] = fine.parent.remain

    data = {
        'obj_id': fine.parent_id or fine.id
    }
    users = [i.user for i in fine.car_detail.cars.only('user')]
    notifications = _create_content(fine, users, action, template_data, data)
    _send_data_notifications(notifications)


def send_report_notification(report: Report, action):
    users = [report.created_by]
    notifications = _create_content(report, users, action)
    _send_data_notifications(notifications)


def send_offence_notification(offence: Offence, action):
    data = {
        'obj_id': offence.id,
        'report_id': offence.report_id
    }
    users = [offence.report.created_by]
    notifications = _create_content(offence, users, action, data=data)
    _send_data_notifications(notifications)


def send_inspections_end_notification(items: QuerySet[Inspection], action):
    notifications = []
    for inspection in items:
        template_data = {
            'dateNextInpsection': str(inspection.dateNextInpsection),
            'pPlateNumber': inspection.car_detail.number,
        }
        data = {
            'dateNextInpsection': str(inspection.dateNextInpsection),
            'pDivision': inspection.pDivision,
            'carDetailId': inspection.car_detail.id
        }
        users = [i.user for i in inspection.car_detail.cars.only('user')]
        notifications.extend(
            _create_content(inspection, users, action, template_data, data)
        )
    _send_data_notifications(notifications)


def send_confidants_end_notification(items: QuerySet[Confidant], action):
    notifications = []
    for confidant in items:
        data = {
            'date_end': str(confidant.date_end),
            'pNumber': confidant.number,
            'pPlateNumber': confidant.car.car_detail.number,
        }
        notifications.extend(
            _create_content(confidant, [confidant.car.user], action, data=data)
        )
    _send_data_notifications(notifications)


def send_insurance_end_notification(items: QuerySet[Insurance], action):
    notifications = []
    for insuranse in items:
        data = {
            'date_end': str(insuranse.date_end),
            'pNumber': insuranse.number,
            'pPlateNumber': insuranse.car.car_detail.number,
        }
        notifications.extend(
            _create_content(insuranse, [insuranse.car.user], action, data=data)
        )
    _send_data_notifications(notifications)


def send_tinting_end_notification(items: QuerySet[Tinting], action):
    notifications = []
    for tinting in items:
        data = {
            'date_end': str(tinting.date_end),
            'pNumber': tinting.number,
            'pPlateNumber': tinting.car.car_detail.number,
        }
        notifications.extend(
            _create_content(tinting, [tinting.car.user], action, data=data)
        )
    _send_data_notifications(notifications)
