# Django
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.conf import settings
from django.db import models

# Pypi: django-parler
from parler.models import TranslatedFields

# Project
from apps.shared.models import BaseTranslateModel, BaseModel


class NotificationTemplate(BaseTranslateModel):
    # Fine
    PARTIAL_PAID_FINE = "partial_paid_fine"
    COMPLETELY_PAID_FINE = "completely_paid_fine"
    NEW_FINE = "new_fine"
    FINE_PAY_SUCCESS = 'pay_fine_success'
    FINE_PAY_FAILED = 'pay_fine_failed'
    FINE_DISCOUNT_ENDING = 'fine_discount_ending'
    FINE_IMAGES_FOUND = 'fine_images_found'

    # Report
    REPORT_REVIEWED = 'report_reviewed'
    OFFENCE_CHANGED_STATUS = 'offence_changed_status'

    # Inspection
    INSPECTION_COMING_END = "inspection_coming_end"

    # Insurance
    INSURANCE_COMING_END = "insurance_coming_end"

    # Tinting
    TINTING_COMING_END = "tinting_coming_end"

    # Confidant
    CONFIDANT_COMING_END = "confidant_coming_end"

    ACTION_CHOICES = (
        (PARTIAL_PAID_FINE, 'Partial paid fine'),
        (COMPLETELY_PAID_FINE, "Completely paid fine"),
        (NEW_FINE, 'New Fine'),
        (FINE_DISCOUNT_ENDING, 'Fine discount ending'),
        (FINE_PAY_SUCCESS, 'Fine pay success'),
        (FINE_PAY_FAILED, 'Fine pay failed'),
        (REPORT_REVIEWED, 'Report reviewed'),
        (OFFENCE_CHANGED_STATUS, 'Offence changed status'),
        (FINE_IMAGES_FOUND, 'Fine images found'),
        (INSPECTION_COMING_END, "Inspection coming end"),
        (INSURANCE_COMING_END, "Insurance coming end"),
        (TINTING_COMING_END, "Tinting coming end"),
        (CONFIDANT_COMING_END, "Confidant coming end"),
    )

    action = models.CharField(
        max_length=50,
        choices=ACTION_CHOICES,
        unique=True
    )

    translations = TranslatedFields(
        title=models.CharField(max_length=100),
        body=models.CharField(max_length=2048)
    )


class NotificationContent(BaseTranslateModel):
    template = models.ForeignKey(
        NotificationTemplate,
        on_delete=models.PROTECT
    )
    translations = TranslatedFields(
        content=models.JSONField(default=dict)
    )

    content_type = models.ForeignKey(
        ContentType,
        on_delete=models.CASCADE,
        null=True
    )
    object_id = models.PositiveIntegerField()
    content_object = GenericForeignKey('content_type', 'object_id')


class Notification(BaseModel):
    notification_content = models.ForeignKey(
        NotificationContent,
        on_delete=models.CASCADE
    )
    user = models.ForeignKey('gauth.User', on_delete=models.CASCADE)
    is_read = models.BooleanField(default=False)
