# Django Rest Framework
from rest_framework import mixins, status
from rest_framework.permissions import IsAuthenticated
from rest_framework.viewsets import GenericViewSet
from rest_framework.decorators import action
from rest_framework.response import Response

# Project
from ..models import Notification
from .serializers import NotificationSerializer, NotificationCountSerializer


class NotificationViewSet(mixins.ListModelMixin,
                          mixins.RetrieveModelMixin,
                          mixins.UpdateModelMixin,
                          mixins.DestroyModelMixin,
                          GenericViewSet):
    permission_classes = (IsAuthenticated,)

    def get_serializer_class(self):
        if self.action == 'count':
            return NotificationCountSerializer
        return NotificationSerializer

    def get_queryset(self):
        queryset = Notification.objects.filter(
            user=self.request.user
        ).select_related('notification_content__template')
        return queryset

    def destroy(self, request, *args, **kwargs):
        instance = self.get_object()
        instance.is_deleted = True
        instance.save()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(methods=['GET'], detail=False)
    def count(self, request, *args, **kwargs):
        queryset = self.get_queryset()
        data = {
            'total_count': queryset.count(),
            'unread_count': queryset.filter(is_read=False).count()
        }
        return Response(self.get_serializer(data).data)
