# Stdlib
import json
from unittest.mock import Mock, PropertyMock

# Django
from django.test import TestCase

# Project
from apps.logs.utils import log_http_output_request, log_http_input_request
from apps.logs.mongo import mongo

db = mongo.log


class LogTest(TestCase):
    def setUp(self) -> None:
        """ Clear data from collections"""
        db.http_input_v2.drop()
        db.http_output.drop()

    def test_input(self):
        collection = db.http_input_v2

        request = Mock()
        request.build_absolute_uri.return_value = 'https://test.com'
        type(request).data = PropertyMock(return_value={'request': 'data'})
        type(request).method = PropertyMock(return_value="GET")

        response = Mock()
        type(response).status_code = PropertyMock(return_value=200)
        type(response).data = PropertyMock(return_value={"Hello": "World"})

        log_http_input_request(request, response)

        log = collection.find_one()
        self.assertEqual(
            log,
            {
                '_id': log['_id'],
                'request_url': 'https://test.com',
                'request_body': {'request': 'data'},
                'request_method': 'GET',
                'response_status_code': 200,
                'response_body': {"Hello": "World"},
                'created_date': log['created_date']
            }
        )

    def test_output(self):
        collection = db.http_output

        response = Mock()
        response.json.return_value = {'hello': 'world'}
        type(response).url = PropertyMock(return_value="https://test.com")
        type(response).status_code = PropertyMock(return_value=200)
        type(response.request).body = PropertyMock(
            return_value=json.dumps({'hello': 'world'}).encode('utf8')
        )
        type(response.request).method = PropertyMock(return_value='GET')
        response.elapsed.total_seconds.return_value = 1

        log_http_output_request(response)

        log = collection.find_one()
        self.assertEqual(
            log,
            {
                '_id': log['_id'],
                'request_url': 'https://test.com',
                'request_body': {'hello': 'world'},
                'request_method': 'GET',
                'response_status_code': 200,
                'response_body': {"hello": "world"},
                'created_date': log['created_date'],
                'response_time': 1
            }
        )
