# Django
from django.utils.translation import get_language

# Django Rest Framework
from rest_framework.validators import ValidationError

# Project
from apps.shared.utils import requests

DOMAIN = 'http://admin:admin@geocoder.road24.uz'


def _prettify(data):
    content = data['features'][0]['properties']['geocoding']

    response = {'country': content['country']}

    if content['country'] not in ['Узбекистан', 'Oʻzbekiston']:
        msg = 'Works only in Uzbekistan.'
        raise ValidationError({'non_field_errors': [msg]})

    if content.get('state'):
        response['region'] = content['state']
        response['district'] = content.get('county') or content.get('district')
    else:
        response['region'] = content['city']
        response['district'] = content.get('district') or content.get('county')

    address = [response['region'], response['district']]

    if street := content.get('street'):
        address.append(street)

    if locality := content.get('locality'):
        address.append(locality)

    if housenumber := content.get('housenumber'):
        address.append(housenumber)

    response['address'] = ', '.join(address)
    return response


def reverse(*, lat: float, lon: float, language=None) -> dict:
    url = f'{DOMAIN}/reverse?lat={lat}&lon={lon}&zoom=18&format=geocodejson'
    headers = {'Accept-Language': language or get_language()}

    response = requests.get(url, headers=headers)
    data = response.json()

    if error := data.get('error'):
        return {'error': error}
    else:
        return _prettify(data)


def search(text: str) -> dict:
    url = f'{DOMAIN}/search?q={text}' \
          f'&format=geocodejson&countrycodes=uz'
    headers = {'Accept-Language': get_language()}
    response = requests.get(url, headers=headers)
    data = response.json()

    if error := data.get('error'):
        return {'error': error}
    else:
        return data
