# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.gauth.models import User
from apps.shared.tests import ViewSetTestCase, AuthTestCase


class UserMobileTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = ['user.yaml']

    url = 'mobile:user-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._retrieve({'method': 'me'})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_me(self):
        response = self._retrieve({'method': 'me'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "phone": "+999999999990",
                "email": "",
                "image": None,
                "gender": "",
                "birthday": None,
                "first_name": "Hello",
                "last_name": "World",
                "language": "ru"
            }
        )

    def test_me_update(self):
        data = {
            'id': 999,
            'first_name': "hello",
            'last_name': "World",
            'phone': '2222'
        }
        response = self._update({'method': 'me'}, data=data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        user = User.objects.get(id=1)
        self.assertEqual(user.phone, '+999999999990')
        self.assertEqual(user.first_name, data['first_name'])
        self.assertEqual(user.last_name, data['last_name'])

    def test_me_update_empty_first_name_last_name(self):
        data = {
            'id': 999,
            'first_name': "",
            'last_name': "",
            'phone': '2222'
        }
        response = self._update({'method': 'me'}, data=data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        user = User.objects.get(id=1)
        self.assertEqual(user.phone, '+999999999990')
        self.assertEqual(user.first_name, "")
        self.assertEqual(user.last_name, "")

    def test_me_update_empty(self):
        data = {}
        response = self._update({'method': 'me'}, data=data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        user = User.objects.get(id=1)
        self.assertEqual(user.phone, '+999999999990')

    def test_user_token_success(self):
        response = self._create(
            {'phone': 999999999991},
            {'method': 'user-token'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(list(response.data.keys()), ['refresh', 'access'])

    def test_user_token_fail_not_found(self):
        response = self._create(
            {'phone': 99999},
            {'method': 'user-token'}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'phone': ['Not found.']})

    def test_user_token_fail_perm(self):
        self._auth_non_admin()
        response = self._create(
            {'phone': 99999},
            {'method': 'user-token'}
        )
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)
