# Stdlib
from datetime import timedelta

# Django
from django.utils import timezone
from django.urls import reverse
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.gauth.models import User
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared.messages import INVALID_MAX_LENGTH, INVALID_MIN_LENGTH


class AuthMobileTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = ['user.yaml']

    url = 'mobile:auth-%s'

    def setUp(self):
        self._auth_admin()

    def test_auth_phone_new_user(self):
        self.assertEqual(User.objects.count(), 4)

        data = {'phone': '998935115012'}
        response = self._create(data, {'method': 'phone'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(User.objects.count(), 5)

        user = User.objects.first()
        self.assertIsNotNone(user.sms_code)
        self.assertTrue(user.is_active)
        self.assertEqual(user.phone, '+' + data['phone'])

    def test_auth_fail_min_length(self):
        data = {'phone': '998'}
        response = self._create(data, {'method': 'phone'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'phone': [INVALID_MIN_LENGTH.format(min_length=12)]}
        )

    def test_auth_fail_max_length(self):
        data = {'phone': '99899999999999123'}
        response = self._create(data, {'method': 'phone'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'phone': [INVALID_MAX_LENGTH.format(max_length=12)]}
        )

    def test_verify_auth_phone_fail_expired(self):
        user = User.objects.get(id=1)
        user.sms_date = timezone.now() - timedelta(minutes=2)
        user.save()

        self.assertEqual(user.sms_code, '123123')
        self.assertEqual(user.sms_date.date(), timezone.now().date())

        data = {
            'phone': '999999999990',
            'sms_code': 123123
        }
        response = self._create(data, {'method': 'phone-verify'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'non_field_errors': [_('Sms code is expired.')]}
        )

    def test_verify_auth_phone_fail_sms_code(self):
        data = {
            'phone': '999999999990',
            'sms_code': 123124
        }
        response = self._create(data, {'method': 'phone-verify'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'non_field_errors': [_('Wrong sms code.')]}
        )

    def _test_access_token(self, access_token):
        url = reverse('mobile:user-me', kwargs={'version': 'v1'})

        self.client.credentials()
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        self.client.credentials(
            HTTP_AUTHORIZATION='JWT ' + access_token
        )
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def _test_refresh_token(self, refresh_token):
        url = reverse('mobile:token_refresh', kwargs={'version': 'v1'})

        self.client.credentials()
        response = self.client.post(url, {'refresh': refresh_token})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertTrue(response.data['access'])
        self.assertTrue(response.data['refresh'])

    def test_verify_auth_phone(self):
        user = User.objects.get(id=1)
        user.sms_date = timezone.now()
        user.save()

        self.assertEqual(user.sms_code, '123123')
        self.assertEqual(user.sms_date.date(), timezone.now().date())

        data = {
            'phone': '999999999990',
            'sms_code': 123123
        }
        response = self._create(data, {'method': 'phone-verify'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertTrue(response.data['refresh'])
        self.assertTrue(response.data['access'])

        user.refresh_from_db()
        self.assertEqual(user.sms_code, "")
        self.assertEqual(user.sms_date.date(), timezone.now().date())

        self._test_access_token(response.data['access'])
        self._test_refresh_token(response.data['refresh'])

    def test_auth_test_account(self):
        data = {'phone': '998917776990'}
        response = self._create(data, {'method': 'phone'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(User.objects.count(), 5)
        user = User.objects.get(phone='+998917776990')
        self.assertEqual(user.sms_code, '111111')
        self.assertEqual(user.sms_date.date(), timezone.now().date())
