# Django
from django.urls import reverse

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.gauth.models import User
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared.messages import INVALID_REQUIRED, INVALID_UNIQUE


class UserAdminTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml'
    ]

    url = 'myadmin:users-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self._unauth()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._update({'pk': 1}, {})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 4)
        self.assertEqual(
            response.data['results'][0],
            {
                "id": 4,
                "is_superuser": False,
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                "modified_date": "2017-01-01T00:00:00.100000+05:00",
                "is_deleted": False,
                "username": "ejarima",
                "phone": None,
                "is_active": True,
                "email": "",
                "gender": "",
                "birthday": None,
                "first_name": "",
                "last_name": "",
                "language": "ru"
            }
        )

    def test_pagination(self):
        response = self._list({'page_size': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 4)
        self.assertEqual(len(response.data['results']), 1)

    def test_retrieve(self):
        response = self._retrieve({'pk': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 2,
                "is_superuser": False,
                "created_date": "2017-01-01T00:00:00.100000+05:00",
                "modified_date": "2017-01-01T00:00:00.100000+05:00",
                "is_deleted": False,
                "username": "lFJlruRRypf0MCpcIoXRj3BFdWG3",
                "phone": "+999999999991",
                "is_active": True,
                "email": "",
                "gender": "",
                "birthday": None,
                "first_name": "",
                "last_name": "",
                "language": "ru"
            }
        )

    def test_create_fail_require(self):
        data = {}
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'username': [INVALID_REQUIRED],
                'password': [INVALID_REQUIRED]
            }
        )

    def test_create_success(self):
        data = {
            'username': 'new_user',
            'password': 'password',
            'phone': '+999 99 999 99 90'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        user = User.objects.get(id=response.data['id'])
        self.assertEqual(user.username, data['username'])

    def test_create_fail_username_unique(self):
        data = {
            'username': 'gOnHiaPt9ZfB8xXpT9dGXtOrXig2',
            'password': 'password'
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'username': ['user с таким username уже существует.']
            }
        )

    def test_update(self):
        data = {
            'username': 'new_user',
            'password': 'password'
        }
        response = self._update({'pk': 1}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        user = User.objects.get(id=1)
        self.assertEqual(user.username, data['username'])

    def test_patch(self):
        data = {
            'first_name': "first_name_first_name"
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        user = User.objects.get(id=1)
        self.assertEqual(user.first_name, data['first_name'])

    def test_delete(self):
        response = self._destroy({'pk': 2})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
