# Stdlib
from datetime import timedelta

# Django
from django.utils import timezone
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework import serializers
from rest_framework_simplejwt.tokens import RefreshToken

# Project
from ...services import auth_phone
from ...models import User


class AuthPhoneSerializer(serializers.Serializer):
    phone = serializers.CharField(
        min_length=12,
        max_length=12
    )

    def validate_phone(self, phone):
        return phone

    def create(self, validated_data):
        try:
            auth_phone(validated_data['phone'])
        except Exception as e:
            raise serializers.ValidationError(e)

        return validated_data


class AuthPhoneVerifySerializer(serializers.Serializer):
    phone = serializers.CharField(
        min_length=12,
        max_length=12
    )
    sms_code = serializers.IntegerField()

    def validate(self, attrs):
        now = timezone.now()
        phone = attrs['phone']
        sms_code = attrs['sms_code']
        user = self.user = User.objects.filter(
            phone='+' + phone,
            sms_code=sms_code
        ).first()

        if not user:
            raise serializers.ValidationError({
                'non_field_errors': [_('Wrong sms code.')]
            })

        if now > (user.sms_date + timedelta(minutes=1)):
            raise serializers.ValidationError({
                'non_field_errors': [_('Sms code is expired.')]
            })

        return attrs

    def create(self, validated_data):
        user = self.user
        refresh = RefreshToken.for_user(user)
        user.sms_code = ""
        user.last_login = timezone.now()
        user.save()

        return {
            'refresh': str(refresh),
            'access': str(refresh.access_token),
        }
