# Django
from django.contrib.auth import get_user_model
from django.utils.translation import gettext_lazy as _

# Django Rest Framework
from rest_framework.authentication import (
    BaseAuthentication, get_authorization_header
)
from rest_framework import exceptions

# Pypi: firebase
from firebase_admin import auth

# Project
from .services import create_user

User = get_user_model()


class FirebaseAuthentication(BaseAuthentication):
    auth_header_prefix = 'Bearer'

    def authenticate(self, request):
        firebase_token = self.get_token(request)

        if not firebase_token:
            return None

        try:
            payload = auth.verify_id_token(firebase_token)
        except ValueError:
            msg = _("Invalid firebase ID token.")
            raise exceptions.AuthenticationFailed(msg)
        except (
                auth.ExpiredIdTokenError,
                auth.InvalidIdTokenError,
                auth.RevokedIdTokenError,
        ):
            msg = _("Could not log in.")
            raise exceptions.AuthenticationFailed(msg)

        user = self.authenticate_credentials(payload)

        return user, payload

    def get_token(self, request):
        """
        Returns the firebase ID token from request.
        """
        auth = get_authorization_header(request).split()

        if not auth or \
                auth[0].lower() != self.auth_header_prefix.lower().encode():
            return None

        if len(auth) == 1:
            msg = _("Invalid Authorization header. No credentials provided.")
            raise exceptions.AuthenticationFailed(msg)
        elif len(auth) > 2:
            msg = _(
                "Invalid Authorization header. Token string "
                "should not contain spaces."
            )
            raise exceptions.AuthenticationFailed(msg)

        return auth[1]

    def authenticate_credentials(self, payload):
        uid = payload['uid']
        phone_number = payload['phone_number']

        user, _ = create_user(phone_number, uid)

        if not user.is_active:
            raise exceptions.AuthenticationFailed(
                _('User inactive or deleted.')
            )

        return user
