# Django
from django.db.models import Value
from django.db.models.functions import Concat

# Django Rest Framework
from rest_framework.viewsets import ModelViewSet
from rest_framework.permissions import IsAdminUser
from rest_framework_simplejwt.tokens import RefreshToken
from rest_framework.authtoken.views import ObtainAuthToken
from rest_framework.response import Response

# Project
from ..models import User
from .serializers import UserAdminSerializer


class CustomAuthToken(ObtainAuthToken):
    """
        `
    API для получения токена для авторизации.
    Токен нужно использовать в виде:
    JWT ca2577d1783362cd3b4c7faf5befed99f20c8d4a
        """

    def post(self, request, *args, **kwargs):
        serializer = self.serializer_class(data=request.data,
                                           context={'request': request})
        serializer.is_valid(raise_exception=True)
        user = serializer.validated_data['user']
        refresh = RefreshToken.for_user(user)

        return Response({
            'refresh': str(refresh),
            'access': str(refresh.access_token)
        })


class UserViewSet(ModelViewSet):
    permission_classes = (IsAdminUser,)
    serializer_class = UserAdminSerializer
    filterset_fields = ['is_superuser']
    search_fields = ['fullname', 'username', 'phone']
    queryset = User.objects.all().annotate(
        fullname=Concat('first_name', Value(' '), 'last_name'))
