# Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.files.models import File
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared.utils import generate_image, generate_video


class FilesPublicTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = ['user.yaml', 'file.yaml']

    url = 'public:files-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._retrieve({"pk": 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        file = generate_image()
        response = self._create(
            {'file': file},
            {'headers': dict(format='multipart')}
        )
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_retrieve(self):
        response = self._retrieve({"pk": 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                'id': 1,
                'name': 'my_file',
                'file': 'http://testserver/media/test.jpg',
                'video_preview': None,
                'module': ''
            }

        )

    def test_create(self):
        self.assertEqual(File.objects.count(), 6)
        file = generate_image()
        response = self._create(
            {'file': file},
            {'headers': dict(format='multipart')}
        )
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertTrue('id' in response.data)
        self.assertEqual(File.objects.count(), 7)

    def test_create_report_video_success(self):
        file = generate_video()
        response = self._create(
            {
                'file': file,
                'module': File.REPORT
            },
            {'headers': dict(format='multipart')}
        )
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        file = File.objects.get(id=response.data['id'])
        self.assertNotEqual(file.video_preview, '')
        self.assertEqual(file.module, File.REPORT)

    def test_create_report_video_fail_duration(self):
        file = generate_video(121)
        response = self._create(
            {
                'file': file,
                'module': File.REPORT
            },
            {'headers': dict(format='multipart')}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'file': ['Max duration 120s.']})

    def test_create_report_video_fail_not_video(self):
        file = generate_image()
        response = self._create(
            {
                'file': file,
                'module': File.REPORT
            },
            {'headers': dict(format='multipart')}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'file': ['Not video.']})
