# Stdlib
import io

# Django
from django.db import transaction
from django.core.files.base import ContentFile

# Django Rest Framework
from rest_framework import serializers

# Pypi: moviepy
from moviepy.editor import VideoFileClip

# Pypi: PIL
from PIL import Image

# Project
from .models import File
from apps.shared.serializers import DynamicFieldsModelSerializer


class FileSerializer(DynamicFieldsModelSerializer):
    class Meta:
        model = File
        fields = ('id', 'name', 'file', 'video_preview', 'module')
        read_only_fields = ('video_preview',)

    @transaction.atomic
    def create(self, validated_data):
        instance: File = super().create(validated_data)

        if instance.module == File.REPORT:
            path = instance.file.path
            try:
                video = VideoFileClip(path)
            except Exception as e:
                raise serializers.ValidationError({
                    'file': ['Not video.']
                })
            if video.duration > 120:
                raise serializers.ValidationError({
                    'file': ['Max duration 120s.']
                })

            frame = video.get_frame(1)
            image = Image.fromarray(frame)

            data = io.BytesIO()
            image.save(data, 'JPEG')
            data.seek(0)

            file = data.getvalue()
            instance.video_preview.save('preview.jpeg', ContentFile(file))

        return instance
