# Django
from django.conf.global_settings import DATETIME_INPUT_FORMATS

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.shared.utils import requests
from ..models import Fine, CarDetail
from ..services.fines import create_fine
from ..services.cars import unsubscribe_to_fines

FORMATS = DATETIME_INPUT_FORMATS + [
    '%d.%m.%Y %H:%M:%S',
    '%H:%M:%S',
    ' %H:%M:%S'
]


class FineSerializer(serializers.ModelSerializer):
    class Meta:
        model = Fine
        fields = (
            'pId',
            'pSeryNumber',
            'pPlace',
            'pLocation',
            'pDate',
            'pViolation',
            'pAmount',
            'pStatus',
            'pUrl'
        )
        extra_kwargs = {
            'pId': {'source': 'car_detail'},
            'pDate': {'input_formats': FORMATS}
        }

    @property
    def errors(self):
        errors = self._errors
        if not errors:
            return {}

        if errors.get('pId'):
            pId = self.initial_data['pId']
            car_detail = CarDetail.raw_objects.filter(id=pId).first()
            if car_detail:
                car_detail.require_unsubscribe = True
                car_detail.save()
            return {
                "code": 8,
                "message": "Car not found"
            }

        return {
            "code": -2,
            "message": errors
        }

    def to_representation(self, instance):
        return instance

    def create(self, validated_data):
        return create_fine(**validated_data)
