# Stdlib
from unittest.mock import patch, Mock

# Django
from django.urls import reverse

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages as msg
from apps.cars.models import Fine, CarDetail
from apps.notifications.models import Notification, NotificationContent


class WebhookFinesTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cars.yaml',
        'car_model.yaml',
        'car_brand.yaml',
        'car_detail.yaml',
        'violation.yaml',
        'fines.yaml',
        'file.yaml',
        'notification_template.yaml'
    ]

    url = 'webhook:fines'

    def setUp(self):
        self._auth_saferoad()

    def test_unauth(self):
        self.client.credentials()

        response = self.client.post(reverse(self.url))
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        self._auth_admin()
        response = self.client.post(reverse(self.url))
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_create_fine(self):
        self.assertEqual(Notification.objects.count(), 0)
        self.assertEqual(NotificationContent.objects.count(), 0)

        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(),
            1
        )
        data = {
            'pId': '3',
            'pSeryNumber': '123',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 1,
            'pUrl': 'asdasd'
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )
        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(),
            2
        )
        fine = Fine.objects.filter(car_detail__cars__id=3).first()
        self.assertEqual(fine.pUrl, data['pUrl'])

        self.assertEqual(Notification.objects.count(), 1)
        self.assertEqual(NotificationContent.objects.count(), 1)
        content = NotificationContent.objects.get()
        self.assertEqual(content.translations.count(), 3)

    @patch('apps.cars.services.fines.refresh_remain', return_value=None)
    def test_create_fine_child_exists_parent(self, _mock_refresh_remain):
        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(),
            1
        )
        data = {
            'pId': '3',
            'pSeryNumber': 'RA20140040418',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 4,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )
        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(),
            2
        )
        self.assertEqual(Fine.objects.get(id=1).child.count(), 1)

    def test_create_child_fine_without_parent(self):
        data = {
            'pId': '4',
            'pSeryNumber': 'RA2014004022',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 4,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )

        fines = Fine.objects.filter(pSeryNumber='RA2014004022')
        self.assertEqual(len(fines), 2)
        for fine in fines:
            self.assertIsNone(fine.parent)
            self.assertEqual(fine.pStatus, 4)

        data['pStatus'] = 1
        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )

        fines_count = Fine.objects.filter(pSeryNumber='RA2014004022').count()
        self.assertEqual(fines_count, 3)

        fines = Fine.objects.filter(
            pSeryNumber='RA2014004022',
            parent__isnull=False
        )
        parent_fine = Fine.objects.get(
            pSeryNumber='RA2014004022',
            parent__isnull=True
        )
        for fine in fines:
            self.assertEqual(fine.parent, parent_fine)
            self.assertEqual(fine.pStatus, 4)

    def test_update_fine(self):
        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(),
            1
        )
        data = {
            'pId': '3',
            'pSeryNumber': 'RA20140040418',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 1,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 0,
                "message": "Success"
            }
        )
        self.assertEqual(
            Fine.objects.filter(car_detail__cars__id=3).count(), 1
        )
        fine = Fine.objects.get(id=1)
        self.assertEqual(fine.pStatus, data['pStatus'])

    def test_create_fine_fail_car_not_found(self):
        data = {
            'pId': 33,
            'pSeryNumber': 'RA20140040418',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 1,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 8,
                "message": "Car not found"
            }
        )

    def test_create_fine_fail_pViolation_not_found(self):
        data = {
            'pId': 1,
            'pSeryNumber': 'RA20140040418',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 333,
            'pAmount': 111500000,
            'pStatus': 1,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": -2,
                "message": {
                    "pViolation": [msg.INVALID_OBJECT404.format(pk_value=333)]
                }
            }
        )

    def test_fine_to_mib(self):
        self.assertEqual(Fine.objects.get(id=3).pStatus, 1)
        data = {
            'pId': 1,
            'pSeryNumber': 'RA20140040420',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 3,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(Fine.objects.get(id=3).pStatus, 3)

    def test_fine_car_not_found_car_unsubscribe(self):
        car_detail = CarDetail.raw_objects.get(id=7)
        self.assertFalse(car_detail.require_unsubscribe)
        self.assertTrue(car_detail.is_deleted)

        data = {
            'pId': 7,
            'pSeryNumber': 'RA20140040420',
            'pPlace': 'ЮҚОРИЧИРЧИҚ ТУМАНИ, 4Р-12 "БЕКТЕМИР-ЧИРЧИК-'
                      'ГАЗАЛКЕНТ-ЧОРВАК" А/Й 16,6-18,5 КМ.ЛАРИ ОРАЛИГИДАГИ '
                      '"ИСТИКЛОЛ" А. Я/П.',
            'pLocation': '',
            'pDate': '14.08.2020 10:33:59',
            'pViolation': 37,
            'pAmount': 111500000,
            'pStatus': 3,
            'pUrl': None
        }

        response = self.client.post(reverse(self.url), data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "code": 8,
                "message": "Car not found"
            }
        )

        car_detail.refresh_from_db()
        self.assertTrue(car_detail.require_unsubscribe)
        self.assertTrue(car_detail.is_deleted)
