# Stdlib
from datetime import datetime, timedelta, date
from unittest.mock import patch, Mock
import os

# Django
from django.conf import settings

# Django Rest Framework
from rest_framework.test import APITestCase

# Pypi: freezegun
from freezegun import freeze_time

# Project
from apps.cars.models import Fine
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.cars.tasks import task_notify_discount_ending, repeat_parse_pdf
from apps.notifications.models import Notification


def mock_pdf_file(file_name, status_code=200):
    path = os.path.join(
        settings.BASE_DIR, f'apps/cars/tests/fine_pdf_files/{file_name}'
    )
    with open(path, 'r') as file:
        mock = Mock()
        mock.status_code = status_code
        mock.text = file.read()
    return mock


class FineTaskTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cars.yaml',
        'car_model.yaml',
        'car_brand.yaml',
        'car_detail.yaml',
        'violation.yaml',
        'violation_translation.yaml',
        'fines.yaml',
        'file.yaml',
        'notification_template.yaml'
    ]

    @freeze_time("2017-02-01")
    def test_task(self):
        now_date = datetime.now().date()
        self.assertEqual(Notification.objects.count(), 0)
        task_notify_discount_ending()

        notifications = Notification.objects.all()
        self.assertEqual(notifications.count(), 2)

        self.assertEqual(
            notifications[0].notification_content.content['body'],
            '%s %s дня' % (str(now_date + timedelta(days=2)), 2)
        )
        self.assertEqual(
            notifications[1].notification_content.content['body'],
            '%s %s дней' % (str(now_date + timedelta(days=7)), 7)
        )

    @patch('apps.cars.tasks.requests.get')
    def test_task_repeat_parse_pdf_success(self, mock_get):
        mock_get.side_effect = [
            mock_pdf_file('pdf_success_2.html'),
            mock_pdf_file('pdf_success_1.html'),
        ]

        self.assertEqual(
            list(Fine.objects.failed_fines().values_list('id', flat=True)),
            [2, 1]
        )
        self.assertEqual(Notification.objects.count(), 0)
        repeat_parse_pdf()
        self.assertEqual(Fine.objects.failed_fines().count(), 0)
        self.assertEqual(Notification.objects.count(), 2)

        self.assertEqual(
            list(
                Fine.objects.filter(id__in=[1, 2]).values(
                    'id',
                    'plate_number',
                    'plate_number_and_car',
                    'date_of_ruling'
                )
            ),
            [
                {
                    'id': 2,
                    'plate_number': 8,
                    'plate_number_and_car': 9,
                    'date_of_ruling': date(2020, 8, 14),
                },
                {
                    'id': 1,
                    'plate_number': 11,
                    'plate_number_and_car': 12,
                    'date_of_ruling': date(2021, 6, 26),
                }
            ]
        )

    @patch('apps.cars.tasks.requests.get')
    def test_task_repeat_parse_pdf_invalid(self, mock_get):
        mock_get.side_effect = [
            mock_pdf_file('pdf_not_found.html'),
            TimeoutError
        ]

        self.assertEqual(
            list(Fine.objects.failed_fines().values_list('id', flat=True)),
            [2, 1]
        )
        self.assertEqual(Notification.objects.count(), 0)
        repeat_parse_pdf()
        self.assertEqual(Fine.objects.failed_fines().count(), 1)
        self.assertEqual(Notification.objects.count(), 0)
