# Stdlib
from unittest.mock import Mock, patch
from copy import deepcopy
from datetime import datetime, timedelta

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Pypi: dateutil
from dateutil import parser, tz

# Pypi: freezegun
from freezegun import freeze_time

# Pypi: requests
from requests.exceptions import HTTPError, ReadTimeout

# Project
from apps.cars.models import CarDetail, Inspection
from apps.cars.services.inspection import check_inspection
from apps.cars.tasks import notify_inspection_ending
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.notifications.models import Notification


class InspectionTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cars.yaml',
        'car_model.yaml',
        'car_brand.yaml',
        'car_detail.yaml',
        'inspections.yaml',
        'file.yaml',
        'notification_template.yaml'
    ]

    url = 'mobile:inspections-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    data = {
        "result": {
            "Vehicle": [
                {
                    "pVehicleId": 25424890,
                    "pPlateNumber": "01J117DB",
                    "pModel": "Cobalt",
                    "pVehicleColor": "БЕЛО ДИМЧАТЫЙ",
                    "pRegistrationDate": "10/13/20 11:28:47 AM",
                    "pDivision": "СИРГАЛИ ТРБ",
                    "pYear": "",
                    "pVehicleType": 2,
                    "pKuzov": "XWBJA69VEKA004982",
                    "pFullWeight": "1620",
                    "pEmptyWeight": "1090",
                    "pMotor": "B15D212182552DJDX0618",
                    "pFuelType": 4,
                    "pSeats": "0",
                    "pStands": "",
                    "pComments": None,
                    "Inspection": {
                        "resultInpsection": "Техник кўрикдан ўтган!",
                        "dateInpsection": "10/13/20 11:28:47 AM",
                        "dateNextInpsection": "10/12/21 12:00:00 AM",
                        "specialMarks": "",
                        "authority": "Иногомов Бегзод Азизович"
                    }
                }
            ],
            "Code": 0,
            "Message": ""
        },
        "error": None
    }

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_success(self, mock_response, _mock_token):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = self.data
        mock_response.return_value = mock

        self.assertEqual(Inspection.objects.count(), 4)
        car_detail = CarDetail.objects.get(id=1)
        check_inspection(car_detail)
        self.assertEqual(Inspection.objects.count(), 5)

        inspection = Inspection.objects.first()
        result = self.data['result']['Vehicle'][0]
        self.assertEqual(inspection.value, result)
        self.assertEqual(inspection.pDivision, result['pDivision'])

        time_zone = tz.gettz('UTC')
        dateInpsection = parser.parse(result['Inspection']['dateInpsection'])
        dateNextInpsection = parser.parse(
            result['Inspection']['dateNextInpsection']
        )

        self.assertEqual(
            inspection.dateInpsection,
            dateInpsection.astimezone(time_zone)
        )
        self.assertEqual(
            inspection.dateNextInpsection,
            dateNextInpsection.astimezone(time_zone)
        )

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_not_found(self, mock_response, _mock_token):
        data = {
            'result': {
                'Vehicle': [],
                'Code': 404,
                'Message': 'Vehicles not found!'
            }, 'error': None
        }

        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = data
        mock_response.return_value = mock

        self.assertEqual(Inspection.objects.count(), 4)
        car_detail = CarDetail.objects.get(id=1)
        check_inspection(car_detail)
        self.assertEqual(Inspection.objects.count(), 4)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_empty_dateNext(self, mock_response, _mock_token):
        data = deepcopy(self.data)
        data['result']['Vehicle'][0]['Inspection']['dateNextInpsection'] = ""

        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = data
        mock_response.return_value = mock

        inspections = list(Inspection.objects.values_list('id', flat=True))
        self.assertEqual(len(inspections), 4)
        car_detail = CarDetail.objects.get(id=1)
        check_inspection(car_detail)
        self.assertEqual(Inspection.objects.count(), 5)

        inspection = Inspection.objects.exclude(id__in=inspections).get()
        self.assertIsNone(inspection.dateNextInpsection)

    @freeze_time("2017-01-01")
    def test_task_notify_end_date_next_inspection(self):
        self.assertEqual(Notification.objects.count(), 0)
        notify_inspection_ending()
        self.assertEqual(Notification.objects.count(), 1)

    @freeze_time("2017-01-20")
    def test_task_notify_end_date_next_inspection_during_14_days(self):
        self.assertEqual(Notification.objects.count(), 0)

        for i in range(40):
            date = (datetime.now() - timedelta(days=i + 1)).date()
            with freeze_time(str(date)):
                notify_inspection_ending()
        self.assertEqual(Notification.objects.count(), 14)

    def test_list_v2(self):
        response = self._list(kwargs={'version': 'v2'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 4,
                    "car_detail": 1,
                    "dateInpsection": "2020-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": None,
                    "pDivision": "Toshkent"
                },
                {
                    "id": 1,
                    "car_detail": 1,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                },
                {
                    "id": 2,
                    "car_detail": 2,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2018-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                }
            ]
        )

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 1,
                    "car_detail": 1,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                },
                {
                    "id": 2,
                    "car_detail": 2,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2018-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                }
            ]
        )

    def test_filter_car_detail(self):
        response = self._list({'car_detail': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 1,
                    "car_detail": 1,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                }
            ]
        )

    def test_pagination(self):
        response = self._list({'page_size': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 1)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_success_new_create(self, mock_response, _mock_token):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = self.data
        mock_response.return_value = mock

        self.assertEqual(Inspection.objects.count(), 4)
        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                'response': self.data,
                'is_new': True
            }
        )
        self.assertEqual(Inspection.objects.count(), 5)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_duplicate_success_request(self, mock_response, _mock_token):
        self.test_check_success_new_create()

        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = self.data
        mock_response.return_value = mock

        self.assertEqual(Inspection.objects.count(), 5)
        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            'response': self.data,
            'is_new': False
        })
        self.assertEqual(Inspection.objects.count(), 5)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_404(self, mock_response, _mock_token):
        data = {
            'result': {
                'Vehicle': [],
                'Code': 404,
                'Message': 'Vehicles not found!'
            },
            'error': None
        }
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = data
        mock_response.return_value = mock

        self.assertEqual(Inspection.objects.count(), 4)
        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            'response': data,
            'is_new': False
        })
        self.assertEqual(Inspection.objects.count(), 4)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_504(self, mock_response, _mock_token):
        mock = Mock()
        mock.status_code = 504
        mock_response.return_value = mock
        mock.raise_for_status.side_effect = [HTTPError]

        self.assertEqual(Inspection.objects.count(), 4)
        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(
            response.status_code, status.HTTP_503_SERVICE_UNAVAILABLE
        )
        self.assertEqual(
            response.data, 'Сервер тех. осмотра временно не работает.'
        )
        self.assertEqual(Inspection.objects.count(), 4)

    @patch('apps.cars.services.inspection._get_token', return_value='token')
    @patch('apps.cars.services.inspection.requests.post')
    def test_check_inspection_read_timeout(self, mock_response, _mock_token):
        mock = Mock()
        mock.status_code = 504
        mock_response.return_value = mock
        mock.raise_for_status.side_effect = [ReadTimeout]

        self.assertEqual(Inspection.objects.count(), 4)
        response = self._create({'car_detail': 1}, {'method': 'check'})
        self.assertEqual(
            response.status_code, status.HTTP_503_SERVICE_UNAVAILABLE
        )
        self.assertEqual(
            response.data, 'Сервер тех. осмотра временно не работает.'
        )
        self.assertEqual(Inspection.objects.count(), 4)

    def test_check_inspection_fail_car(self):
        self.assertEqual(Inspection.objects.count(), 4)
        response = self._create({'car_detail': 3}, {'method': 'check'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'car_detail': ['Not found.']})
