# Stdlib
from unittest.mock import patch, Mock

# Django
from django.db import IntegrityError
from django.urls import reverse
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Pypi: requests
from requests.exceptions import ReadTimeout

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.cars.models import Fine
from apps.transactions.models import Check
from apps.transactions.mobile.serializers import CheckSerializer


class FineMobileTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cars.yaml',
        'car_model.yaml',
        'car_brand.yaml',
        'car_model.yaml',
        'car_brand.yaml',
        'car_detail.yaml',
        'violation.yaml',
        'violation_translation.yaml',
        'fines.yaml',
        'file.yaml',
        'cards.yaml',
        'check.yaml',
    ]

    url = 'mobile:fines-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_constrain_unique_parent_fine(self):
        parent_fine = Fine.objects.create(
            pSeryNumber="9999",
            pDate="2017-01-01 00:00:00",
            pViolation_id=1,
            pAmount="9999",
            pStatus=1,
            car_detail_id=1
        )
        for _ in range(10):
            parent_fine = Fine.objects.create(
                pSeryNumber="9999",
                pDate="2017-01-01 00:00:00",
                pViolation_id=1,
                pAmount="9999",
                pStatus=4,
                car_detail_id=1,
                parent=parent_fine
            )

        error = ''
        try:
            Fine.objects.create(
                pSeryNumber="9999",
                pDate="2017-01-01 00:00:00",
                pViolation_id=1,
                pAmount="9999",
                pStatus=3,
                car_detail_id=1
            )
        except IntegrityError as e:
            error = str(e).replace('\n', ' ')
        except Exception as e:
            assert False

        self.assertEqual(
            error,
            'duplicate key value violates unique constraint '
            '"unique_parent_fine" DETAIL:  Key ("pSeryNumber")=(9999) '
            'already exists. '
        )

    def test_retrieve(self):
        response = self._retrieve({'pk': 3})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 3,
            "pSeryNumber": "RA20140040420",
            "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
            "pLocation": "",
            "pDate": "2017-01-01T00:00:00.100000+05:00",
            "pViolation": {
                "id": 1,
                "desc": "Езда без ременя безопасности",
                "article": "125",
                "part": 1
            },
            "pAmount": "1000.00",
            "pStatus": 1,
            "pUrl": None,
            'pdf': None,
            'plate_number': None,
            'plate_number_and_car': None,
            "pays": [
                {
                    "id": 6,
                    "pAmount": "100.00",
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 5,
                    "pAmount": "200.00",
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                },
                {
                    "id": 4,
                    "pAmount": "400.00",
                    "created_date": "2017-01-01T00:00:00.100000+05:00"
                }
            ],
            "remain": "1000.00",
            "discount_and_paid": {
                'discount': 0,
                'paid': 0.00
            },
            'car': {
                'id': 1
            },
            "date_of_ruling": "2017-01-09"
        })

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        self.assertEqual(
            response.data['results'][0], {
                "id": 7,
                "pSeryNumber": "RA20140040450",
                "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
                "pLocation": "",
                "remain": "800.00",
                "pAmount": "1000.00",
                "pViolation": {
                    "id": 20,
                    "desc": "Установлены дополнительные предметы или нанесены "
                            "покрытия, ограничивающие обзорность с места "
                            "водителя",
                    "article": "126",
                    "part": 1
                },
                "pDate": "2017-01-01T00:00:00.100000+05:00",
                "pStatus": 5,
                "plate_number_and_car": None
            })

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 2)

    def test_filter_car(self):
        response = self._list({'car': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)
        self.assertEqual(
            response.data['results'][0], {
                "id": 3,
                "pSeryNumber": "RA20140040420",
                "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
                "remain": "1000.00",
                "pAmount": "1000.00",
                "pLocation": "",
                "pViolation": {
                    "id": 1,
                    "desc": "Езда без ременя безопасности",
                    "article": "125",
                    "part": 1
                },
                "pDate": "2017-01-01T00:00:00.100000+05:00",
                "pStatus": 1,
                "plate_number_and_car": None
            })

    def test_filter_status(self):
        response = self._list({'pStatus': 5})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 1)
        self.assertEqual(len(response.data['results']), 1)

        self.assertEqual(response.data['results'][0], {
            "id": 7,
            "pSeryNumber": "RA20140040450",
            "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
            "pLocation": "",
            "remain": "800.00",
            "pAmount": "1000.00",
            "pViolation": {
                "id": 20,
                "desc": "Установлены дополнительные предметы или нанесены "
                        "покрытия, ограничивающие обзорность с места водителя",
                "article": "126",
                "part": 1
            },
            "pDate": "2017-01-01T00:00:00.100000+05:00",
            "pStatus": 5,
            "plate_number_and_car": None
        })

    def test_filter_multi_status(self):
        response = self._list({'pStatus': '1,5'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)

        self.assertEqual(
            response.data['results'],
            [
                {
                    "id": 7,
                    "pSeryNumber": "RA20140040450",
                    "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
                    "pLocation": "",
                    "pViolation": {
                        "id": 20,
                        "desc": "Установлены дополнительные предметы или "
                                "нанесены покрытия, ограничивающие обзорность "
                                "с места водителя",
                        "article": "126",
                        "part": 1
                    },
                    "pDate": "2017-01-01T00:00:00.100000+05:00",
                    "pStatus": 5,
                    "plate_number_and_car": None,
                    "remain": "800.00",
                    "pAmount": "1000.00"
                },
                {
                    "id": 3,
                    "pSeryNumber": "RA20140040420",
                    "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
                    "pLocation": "",
                    "pViolation": {
                        "id": 1,
                        "desc": "Езда без ременя безопасности",
                        "article": "125",
                        "part": 1
                    },
                    "pDate": "2017-01-01T00:00:00.100000+05:00",
                    "pStatus": 1,
                    "plate_number_and_car": None,
                    "remain": "1000.00",
                    "pAmount": "1000.00"
                },
                {
                    "id": 2,
                    "pSeryNumber": "RA20140040419",
                    "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
                    "pLocation": "",
                    "pViolation": {
                        "id": 40,
                        "desc": "Проезд водителями транспортных средств на "
                                "запрещающий сигнал светофора или на "
                                "запрещающий жест регулировщика дорожного "
                                "движения",
                        "article": "128X4",
                        "part": 2
                    },
                    "pDate": "2017-01-01T00:00:00.100000+05:00",
                    "pStatus": 1,
                    "plate_number_and_car": None,
                    "remain": "10000000.00",
                    "pAmount": "1000.00"
                }
            ]
        )

    def test_pay_fail_bank_card(self):
        response = self._create({
            'bank_card': 4,
            'amount': 1001
        }, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'bank_card': ['Not found.']
        })

    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000}]
    )
    def test_pay_fail_min_amount(self, _mock):
        response = self._create({
            'bank_card': 1,
            'amount': 999
        }, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            'amount': ['Убедитесь, что это значение больше либо равно 1000.']
        })

    @patch('apps.transactions.services.os.environ', return_value='token')
    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000}]
    )
    @patch('apps.cars.services.fines.get_fine_check')
    @patch('apps.transactions.services.requests.post')
    def test_pay_success(self, mock_request_post,
                         mock_get_fine_check,
                         _mock_get_bank_card_detail,
                         _mock_token):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {
            'status': True,
            'result': {
                "tr_id": "9d5f98d0-87a4-45ff-99ff-095efa93eac1",
                "state": "0",
                "payee": {
                    "account": "4014228601272393430174179",
                    "branch": "00014",
                    "amount": "23400000",
                    "currency": "000"
                },
                "payer": {
                    "name": "MUSLIMOV SANJAR RIXSIBOYEVICH"
                },
                "info": {
                    "purpose": {
                        "code": "08201",
                        "text": "08201~401422863934YEVICH RA20140040418"
                    }
                }
            }
        }

        mock_get_fine_check.return_value = mock

        mock_post = Mock()
        mock_post.status_code = 200
        mock_post.json.return_value = {
            'status': True,
            'result': {
                "tr_id": "9d5f98d0-87a4-45ff-99ff-095efa93eac1",
                "state": "4",
                "payee": {
                    "account": "4014228601272393430174179",
                    "branch": "00014",
                    "inn": "201122919"
                },
                "payer": {
                    "name": "MUSLIMOV SANJAR RIXSIBOYEVICH",
                    "card": {
                        "number": "8600572971603993",
                        "expire": "0824"
                    },
                    "amount": "1000",
                    "commission": "10",
                    "currency": "000"
                },
                "info": {
                    "bid": "KJC207Z3F2129FJI1719",
                    "sid": "2497109885",
                    "tid": "21207563089171958",
                    "confirm": {
                        "code": 0,
                        "message": "Успешно"
                    },
                    "document": {
                        "number": "1415403989",
                        "date": "26.07.2021"
                    },
                    "purpose": {
                        "code": "08201",
                        "text": "08201~401XSIBOYEVICH RA20140040418"
                    }
                }
            }
        }

        mock_request_post.return_value = mock_post

        response = self._create({
            'bank_card': 1,
            'amount': 1001
        }, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        check = CheckSerializer(Check.objects.filter(fine_id=2).first()).data

        self.assertEqual(
            check, {
                "id": 3,
                "created_date": check['created_date'],
                "car": {
                    "number": "01A123AA",
                    "mark": "Tesla1",
                    "name": "Krutaya tachka1"
                },
                "bank_card": {
                    "number": "8600123412341234",
                    "full_name": "Elon Musk"
                },
                "fine": {
                    "pSeryNumber": "RA20140040419",
                    "remain": "234000.00"
                },
                "amount": "1000.00",
                "commission": "10.00",
                "payee_details": {
                    "inn": "201122919",
                    "branch": "00014",
                    "account": "4014228601272393430174179"
                },
                "purpose_text": "08201~401XSIBOYEVICH RA20140040418"
            }
        )

    @patch('apps.cars.mobile.serializers.fine.randint', return_value=123123)
    @patch('apps.cars.mobile.serializers.fine.pay_fine', return_value=None)
    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000000}]
    )
    def test_pay_v2_success(self, *args):
        data = {
            'bank_card': 1,
            'amount': 50000
        }
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        data['sms_code'] = '123123'
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000000}]
    )
    def test_pay_v2_wrong_sms(self, *args):
        data = {
            'bank_card': 1,
            'amount': 50000,
            'sms_code': '333'
        }
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'sms_code': [_('Wrong sms code.')]})

    def test_payments_list(self):
        url = reverse('mobile:payments-list', kwargs={'version': 'v1'})
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 5)
        self.assertEqual(len(response.data['results']), 5)
        self.assertEqual(response.data['results'][-1], {
            "id": 4,
            "car_detail": {
                "id": 1,
                "number": "01A123AA",
                "tech_pass_series": "AAA",
                "tech_pass_num": "123454"
            },
            "pSeryNumber": "",
            "pPlace": "ЮҚОРИЧИРЧИҚ ТУМАНИ 4Р-",
            "pLocation": "",
            "pViolation": {
                "id": 20,
                "desc": "Установлены дополнительные предметы или нанесены "
                        "покрытия, ограничивающие обзорность с места водителя",
                "article": "126",
                "part": 1
            },
            "pDate": "2017-01-01T00:00:00.100000+05:00",
            "pStatus": 4,
            "remain": "1000.00",
            "pAmount": "400.00",
            'parent_id': 3,
            "check_id": 1
        })

    @patch(
        'apps.cars.mobile.views.fine.refresh_remain',
        return_value=[1, {'data': 'data'}]
    )
    def test_refresh_remain(self, _):
        response = self._retrieve({'pk': 7, 'method': 'refresh-remain'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {'data': 'data'})

    @patch('apps.cars.mobile.serializers.fine.randint', return_value=123123)
    @patch('apps.cars.mobile.serializers.fine.pay_fine', return_value=None)
    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000000}]
    )
    def test_pay_v2_duplicate_confirm(self, *args):
        data = {
            'bank_card': 1,
            'amount': 50000
        }
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        data['sms_code'] = '123123'
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'sms_code': ['Неверный смс код']})

    @patch('apps.cars.mobile.serializers.fine.randint', return_value=123123)
    @patch(
        'apps.cars.mobile.serializers.fine.get_bank_card_detail',
        return_value=[True, {'balance': 1000000000}]
    )
    @patch('apps.cars.services.fines.refresh_remain')
    def test_pay_v2_failed(self, mock_refresh_remain, *args):
        mock_refresh_remain.return_value = Fine.objects.get(id=2), {
            'error': {
                'code': '10134',
                'message': 'Ошибка! Приём платежей по данным '
                           'штрафам приостановлен'
            },
            'host': {
                'host': 'UniSoft',
                'time_stamp': '2021-10-31 20:53:31'
            },
            'id': None,
            'jsonrpc': '2.0',
            'origin': 'munis.get.blank',
            'status': False
        }
        data = {
            'bank_card': 1,
            'amount': 50000
        }
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        data['sms_code'] = '123123'
        response = self._create(
            data,
            {'pk': 2, 'method': 'pay', 'version': 'v2'}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'non_field_errors': [
                    'Ошибка! Приём платежей по данным штрафам приостановлен'
                ]
            }
        )

    @patch(
        'apps.bank_cards.utils.requests.post',
        side_effect=[ReadTimeout]
    )
    @patch('os.environ', {'MUNIS_TOKEN': 1})
    def test_pay_fail_bank_timeout(self, *args):
        data = {
            'bank_card': 1,
            'amount': 50000,
        }
        response = self._create(data, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'bank_card': ['На стороне банка не поладки (timeout).']}
        )

    @patch('os.environ', {'MUNIS_TOKEN': 1})
    @patch('apps.bank_cards.utils.requests.post')
    def test_pay_fail_bank_status_500(self, _mock_request, *args):
        mock = Mock()
        mock.status_code = 500
        data = {
            'bank_card': 1,
            'amount': 50000,
        }
        response = self._create(data, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'bank_card': ['На стороне банка не поладки (500).']}
        )

    @patch('os.environ', {'MUNIS_TOKEN': 1})
    @patch('apps.bank_cards.utils.requests.post')
    def test_pay_fail_bank_balance(self, _mock_request, *args):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {
            'status': True,
            'result': {'balance': 1, 'state': 0}
        }
        _mock_request.return_value = mock

        data = {
            'bank_card': 1,
            'amount': 50000,
        }
        response = self._create(data, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'bank_card': [_('Not enough money on card.')]}
        )

    @patch('os.environ', {'MUNIS_TOKEN': 1})
    @patch('apps.bank_cards.utils.requests.post')
    def test_pay_fail_bank_error_message(self, _mock_request, *args):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'status': False, 'error': {'message': {
            'uz': "Uz error",
            'ru': "Ru error",
        }}}
        _mock_request.return_value = mock

        data = {
            'bank_card': 1,
            'amount': 50000,
        }
        response = self._create(data, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {"bank_card": ["Ru error"]})

    @patch('os.environ', {'MUNIS_TOKEN': 1})
    @patch('apps.bank_cards.utils.requests.post')
    def test_pay_fail_bank_card_inactive(self, _mock_request, *args):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'status': True, 'result': {'state': -1}}
        _mock_request.return_value = mock

        data = {
            'bank_card': 1,
            'amount': 50000,
        }
        response = self._create(data, {'pk': 2, 'method': 'pay'})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {"bank_card": ["Карта заблокированна."]}
        )
