# Stdlib
from unittest.mock import patch, Mock, call

# Django
from django.db import IntegrityError
from django.utils.translation import gettext

# Django Rest Framework
from rest_framework.test import APITransactionTestCase
from rest_framework import status

# Pypi: requests
from requests.exceptions import ReadTimeout

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages
from ..errors import TEMPORARLY_UNAVAILABLE
from ..models import Car, Confidant, Insurance, Tinting, CarDetail
from ..tasks import unsubscribe_cars


class CarTest(APITransactionTestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cars.yaml',
        'car_brand.yaml',
        'car_model.yaml',
        'car_detail.yaml',
        'confidant.yaml',
        'inspection_places.yaml',
        'old_inspections.yaml',
        'insurance.yaml',
        'tinting.yaml',
        'oil.yaml',
        'violation.yaml',
        'violation_translation.yaml',
        'fines.yaml',
        'file.yaml',
        'inspections.yaml'
    ]

    url = 'mobile:cars-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 1,
            "number": "01A123AA",
            "tech_pass_num": "123454",
            "tech_pass_series": "AAA",
            "mark": "Tesla1",
            "name": "Krutaya tachka1",
            'image': None,
            "confidant": {
                "id": 4,
                "kind": Confidant.SPECIAL,
                "series": "ART",
                "number": "12345678910",
                "date_passed": "2017-01-01",
                "date_end": "2040-01-01",
                "issued_to": "First, Second, Third"
            },
            "inspection": {
                "id": 4,
                "place": {
                    "id": 3,
                    "name": "Yunusobod"
                },
                "date_passed": "2017-01-01",
                "date_end": "2047-01-01",
            },
            "insurance": {
                "id": 4,
                "kind": Insurance.OSAGO,
                "number_of_drivers": 1,
                "company_name": "insurance_company",
                "series": "ART",
                "number": "12345678910",
                "date_passed": "2017-01-01",
                "date_end": "2040-01-01"
            },
            "tinting": {
                "id": 4,
                "kind": Tinting.BACK,
                "series": "ART",
                "number": "12345678910",
                "date_passed": "2017-01-01",
                "date_end": "2040-01-01",
            },
            "oil": {
                "id": 4,
                "mark": "Castrol",
                "mileage": 14000,
                "next_mileage": 24000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2023-01-01",
                "daily_mileage": None,
                "oil_mileage": None
            },
            'car_model': {
                'id': 1,
                "car_brand": 1,
                'name': 'Nexia',
                'image': {
                    'id': 1,
                    'name': 'my_file',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            'car_brand': {
                'id': 1,
                'name': 'Chevrolet',
                'logo': {
                    'id': 1,
                    'name': 'my_file',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            'fines_count': 2,
            'fines_remain': 10001000.00,
            "car_detail": 1,
            'date_of_registration': None,
            'year_manufacture': None,
            'pinned': False
        })

    def test_retrieve_no_fines(self):
        response = self._retrieve({'pk': 4})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['fines_remain'], 0)
        self.assertEqual(response.data['fines_count'], 0)

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        self.assertEqual(
            response.data['results'][1],
            {
                "id": 2,
                "number": "01A124AA",
                "tech_pass_num": "123454",
                "tech_pass_series": "AAA",
                "mark": "Tesla2",
                "name": "Krutaya tachka2",
                'image': None,
                "confidant": {
                    "id": 2,
                    "kind": Confidant.ONE_TIME,
                    "series": "ART",
                    "number": "12345678911",
                    "date_passed": "2017-01-01",
                    "date_end": "2017-01-01",
                    "issued_to": "First"
                },
                "inspection": {
                    "id": 2,
                    "place": {
                        "id": 2,
                        "name": "Keles"
                    },
                    "date_passed": "2017-01-01",
                    "date_end": "2047-01-01",
                },
                "insurance": {
                    "id": 2,
                    "kind": Insurance.KASKO,
                    'number_of_drivers': 5,
                    'company_name': "insurance_company",
                    "series": "ART",
                    "number": "12345678911",
                    "date_passed": "2017-01-01",
                    "date_end": "2030-01-01",
                },
                "tinting": {
                    "id": 2,
                    "kind": Tinting.BACK,
                    "series": "ART",
                    "number": "12345678911",
                    "date_passed": "2017-01-01",
                    "date_end": "2017-01-01",
                },
                "oil": {
                    "id": 2,
                    "mark": "Rust",
                    "mileage": 12000,
                    "next_mileage": 22000,
                    "charge_price": 1000000,
                    "last_replacement": "2020-01-01",
                    "next_replacement": "2022-01-01",
                    "daily_mileage": None,
                    "oil_mileage": None
                },
                'car_model': {
                    'id': 2,
                    "car_brand": 2,
                    'name': 'Matiz',
                    'image': {
                        'id': 2,
                        'name': 'my_file2',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                'car_brand': {
                    'id': 2,
                    'name': 'Ravon',
                    'logo': {
                        'id': 2,
                        'name': 'my_file2',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                "fines_count": 0,
                "fines_remain": 0,
                "car_detail": 2,
                'date_of_registration': None,
                'year_manufacture': None,
                'pinned': False
            }
        )

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 2)

    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create_fail_saferoad_status_400(self, mock_subscribe_to_fines):
        mock = Mock()
        mock.status_code = 400
        mock.json.return_value = {'code': 0}
        mock_subscribe_to_fines.return_value = mock

        self.assertEqual(CarDetail.objects.count(), 4)
        data = {
            "number": "01A127AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
            "confidant": {
                "kind": Confidant.ONE_TIME,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "issued_to": "First"
            },
            "inspection": {
                "place": 1,
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "kind": Insurance.KASKO,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 4,
                "company_name": "insurance_company"
            },
            "tinting": {
                "kind": Tinting.SIDE,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "oil": {
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01"
            }
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'non_field_errors': ['Saferoad down.']}
        )
        self.assertEqual(CarDetail.objects.count(), 5)

    @patch(
        'apps.cars.services.cars.subscribe_to_fines',
        side_effect=[ReadTimeout]
    )
    def test_create_fail_saferoad_read_timeout(self, _):
        car_details = list(CarDetail.objects.values_list('id', flat=True))
        self.assertEqual(len(car_details), 4)

        data = {
            "number": "01A1337AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'non_field_errors': [
                    gettext(
                        'Error saferoad registration: %s'
                    ) % 'Saferoad timeout'
                ]
            }
        )
        self.assertEqual(CarDetail.objects.count(), 5)
        car_detail = CarDetail.objects.exclude(id__in=car_details).first()
        self.assertFalse(car_detail.is_subscribed)
        self.assertTrue(car_detail.require_unsubscribe)
        self.assertFalse(car_detail.is_deleted)

    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create(self, mock_subscribe_to_fines):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': 0}
        mock_subscribe_to_fines.return_value = mock

        self.assertEqual(CarDetail.objects.count(), 4)
        data = {
            "number": "01A127AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
            'image': 1,
            "confidant": {
                "kind": Confidant.ONE_TIME,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "issued_to": "First"
            },
            "inspection": {
                "place": 1,
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "kind": Insurance.KASKO,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 4,
                "company_name": "insurance_company"
            },
            "tinting": {
                "kind": Tinting.SIDE,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "oil": {
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01",
                "daily_mileage": 100,
                "oil_mileage": None
            },
            'car_brand': 1,
            'car_model': 1
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(response.data, {
            "id": response.data['id'],
            "number": "01A127AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
            'image': {
                'id': 1,
                'name': 'my_file',
                'file': 'http://testserver/media/test.jpg'
            },
            "confidant": {
                "id": response.data['confidant']['id'],
                "kind": Confidant.ONE_TIME,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "issued_to": "First"
            },
            "inspection": {
                "id": response.data['inspection']['id'],
                "place": {
                    "id": 1,
                    "name": "Mirobod"
                },
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "id": response.data['insurance']['id'],
                "kind": Insurance.KASKO,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 4,
                "company_name": "insurance_company"
            },
            "tinting": {
                "id": response.data['tinting']['id'],
                "kind": Tinting.SIDE,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "oil": {
                "id": response.data['oil']['id'],
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01",
                "daily_mileage": 100,
                "oil_mileage": None
            },
            "car_model": {
                "id": 1,
                "car_brand": 1,
                "name": "Nexia",
                "image": {
                    'id': 1,
                    'name': 'my_file',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            "car_brand": {
                "id": 1,
                "name": "Chevrolet",
                "logo": {
                    'id': 1,
                    'name': 'my_file',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            "car_detail": response.data['car_detail'],
            'date_of_registration': None,
            'year_manufacture': None,
            'pinned': False
        })
        self.assertEqual(CarDetail.objects.count(), 5)
        self.assertEqual(
            CarDetail.objects.filter(is_subscribed=True).count(), 5
        )

    def test_create_fail_already_create(self):
        data = {
            "number": "01A123AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "123454",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
            'image': None,
            "confidant": {
                "kind": Confidant.ONE_TIME,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "issued_to": "First"
            },
            "inspection": {
                "place": 1,
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "kind": Insurance.KASKO,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 4,
                "company_name": "insurance_company"
            },
            "tinting": {
                "kind": Tinting.SIDE,
                "series": "AA",
                "number": "12345",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "oil": {
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01"
            }
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'number': [gettext('Already exists.')]}
        )

    @patch(
        'apps.cars.services.cars.Car.objects.create',
        side_effect=[IntegrityError]
    )
    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create_fail_already_create_check_car_detail(
            self, mock_subscribe_to_fines, *_
    ):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': 0}
        mock_subscribe_to_fines.return_value = mock

        car_details = list(CarDetail.objects.values_list('id', flat=True))
        self.assertEqual(len(car_details), 4)
        data = {
            "number": "01A129AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "123454",
            "mark": "Tesla2",
            "name": "Krutaya tachka2"
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'number': [gettext('Already exists.')]}
        )
        self.assertEqual(CarDetail.objects.count(), 5)

        car_detail = CarDetail.objects.exclude(id__in=car_details).get()
        self.assertTrue(car_detail.is_subscribed)
        self.assertFalse(car_detail.is_deleted)

    def test_update_partial(self):
        data = {
            "number": "123454AD",
            "insurance": {
                "id": 4,
                "company_name": "insurance_company3"
            },
            "tinting": {
                "id": 4,
                "kind": Tinting.SIDE
            },
            "oil": {
                "id": 4,
                "mileage": 1000
            }
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['number'], '01A123AA')
        self.assertEqual(
            response.data['insurance']['company_name'],
            data['insurance']['company_name']
        )
        self.assertEqual(response.data['tinting']['kind'], Tinting.SIDE)
        self.assertEqual(response.data['oil']['mileage'], 1000)

    def test_update(self):
        data = {
            "number": "01A124AD",
            "tech_pass_num": "123454",
            "tech_pass_series": "ABC",
            "mark": "Tesla3",
            "name": "Krutaya tachka2",
            'image': None,
            "confidant": {
                "car": 2,
                "kind": Confidant.ONE_TIME,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01",
                "issued_to": "First"
            },
            "inspection": {
                "place": 1,
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "kind": Insurance.OSAGO,
                "series": "AB",
                "number": "123455",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 8,
                "company_name": "insurance_company3"
            },
            "tinting": {
                "car": 2,
                "kind": Tinting.BACK,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01"
            },
            "oil": {
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01",
                "daily_mileage": 100,
                "oil_mileage": None
            }
        }
        response = self._update({'pk': 2}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 2,
            "number": "01A124AA",
            "tech_pass_num": "123454",
            "tech_pass_series": "AAA",
            "mark": "Tesla3",
            "name": "Krutaya tachka2",
            'image': None,
            "confidant": {
                "id": 2,
                "kind": Confidant.ONE_TIME,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01",
                "issued_to": "First"
            },
            "inspection": {
                "id": 2,
                "place": {
                    "id": 1,
                    "name": "Mirobod"
                },
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "id": response.data['insurance']['id'],
                "kind": Insurance.OSAGO,
                "series": "AB",
                "number": "123455",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 8,
                "company_name": "insurance_company3"
            },
            "tinting": {
                "id": 2,
                "kind": Tinting.BACK,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01"
            },
            "oil": {
                "id": response.data['oil']['id'],
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01",
                "daily_mileage": 100,
                "oil_mileage": None
            },
            'car_model': {
                'id': 2,
                "car_brand": 2,
                'name': 'Matiz',
                'image': {
                    'id': 2,
                    'name': 'my_file2',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            'car_brand': {
                'id': 2,
                'name': 'Ravon',
                'logo': {
                    'id': 2,
                    'name': 'my_file2',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            "car_detail": 2,
            'date_of_registration': None,
            'year_manufacture': None,
            'pinned': False
        })

    def test_update_not_found(self):
        response = self._update({'pk': 3})
        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)

    def test_destroy(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        car = Car.raw_objects.get(id=1)
        self.assertEqual(car.is_deleted, True)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)

    def test_update_check_submodules_create(self):
        data = {
            "number": "01A124AD",
            "tech_pass_num": "123454",
            "tech_pass_series": "ABC",
            "mark": "Tesla3",
            "name": "Krutaya tachka2",
            'image': None,
            "confidant": {
                "car": 2,
                "kind": Confidant.ONE_TIME,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01",
                "issued_to": "First"
            },
            "inspection": {
                "place": 1,
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "kind": Insurance.OSAGO,
                "series": "AB",
                "number": "123455",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 3,
                "company_name": "insurance_company3"
            },
            "tinting": {
                "car": 2,
                "kind": Tinting.SIDE,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01"
            },
            "oil": {
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01"
            }
        }
        car = Car.objects.get(id=4)
        self.assertIsNone(car.inspection)
        self.assertIsNone(car.confidant)
        self.assertIsNone(car.insurance)
        self.assertIsNone(car.inspection)
        self.assertIsNone(car.oil)

        response = self._update({'pk': 4}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 4,
            "number": "01J117DB",
            "tech_pass_num": "0476916",
            "tech_pass_series": "AAF",
            "mark": "Tesla3",
            "name": "Krutaya tachka2",
            'image': None,
            "confidant": {
                "id": response.data['confidant']['id'],
                "kind": Confidant.ONE_TIME,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01",
                "issued_to": "First"
            },
            "inspection": {
                "id": response.data['inspection']['id'],
                "place": {
                    "id": 1,
                    "name": "Mirobod"
                },
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11"
            },
            "insurance": {
                "id": response.data['insurance']['id'],
                "kind": Insurance.OSAGO,
                "series": "AB",
                "number": "123455",
                "date_passed": "2020-11-11",
                "date_end": "2040-11-11",
                "number_of_drivers": 3,
                "company_name": "insurance_company3"
            },
            "tinting": {
                "id": response.data['confidant']['id'],
                "kind": Tinting.SIDE,
                "series": "ARTADF",
                "number": "1234567891",
                "date_passed": "2017-01-01",
                "date_end": "2017-01-01"
            },
            "oil": {
                "id": response.data['oil']['id'],
                "mark": "Something",
                "mileage": 12000,
                "next_mileage": 22000,
                "charge_price": 1000000,
                "last_replacement": "2020-01-01",
                "next_replacement": "2022-01-01",
                "daily_mileage": None,
                "oil_mileage": None
            },
            "car_model": {
                "id": 4,
                "car_brand": 1,
                "name": "Lacetti",
                "image": {
                    'id': 3,
                    'name': 'my_file3',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            "car_brand": {
                "id": 1,
                "name": "Chevrolet",
                "logo": {
                    'id': 1,
                    'name': 'my_file',
                    'file': 'http://testserver/media/test.jpg'
                }
            },
            "car_detail": 4,
            'date_of_registration': None,
            'year_manufacture': None,
            'pinned': False
        })

    def test_update_bug_tech_pass_series(self):
        response = self._update({'pk': 1}, {'tech_pass_series': '1234'}, True)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'tech_pass_series': [
                    messages.INVALID_MAX_LENGTH.format(max_length=3)
                ]
            }
        )

    def test_retrieve_v2(self):
        response = self._retrieve({'pk': 1, 'version': 'v2'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "number": "01A123AA",
                "tech_pass_num": "123454",
                "tech_pass_series": "AAA",
                "mark": "Tesla1",
                "name": "Krutaya tachka1",
                'image': None,
                "confidant": {
                    "id": 4,
                    "kind": "special",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01",
                    "issued_to": "First, Second, Third"
                },
                "inspection": {
                    "id": 1,
                    "car_detail": 1,
                    "dateInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": "2017-01-01T00:00:00.100000+05:00",
                    "pDivision": "Toshkent"
                },
                "insurance": {
                    "id": 4,
                    "kind": "osago",
                    "number_of_drivers": 1,
                    "company_name": "insurance_company",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01"
                },
                "tinting": {
                    "id": 4,
                    "kind": "back",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01"
                },
                "oil": {
                    "id": 4,
                    "mark": "Castrol",
                    "mileage": 14000,
                    "next_mileage": 24000,
                    "charge_price": 1000000,
                    "last_replacement": "2020-01-01",
                    "next_replacement": "2023-01-01",
                    "daily_mileage": None,
                    "oil_mileage": None
                },
                'car_model': {
                    'id': 1,
                    "car_brand": 1,
                    'name': 'Nexia',
                    'image': {
                        'id': 1,
                        'name': 'my_file',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                'car_brand': {
                    'id': 1,
                    'name': 'Chevrolet',
                    'logo': {
                        'id': 1,
                        'name': 'my_file',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                "fines_count": 2,
                "fines_remain": 10001000.00,
                "car_detail": 1,
                'date_of_registration': None,
                'year_manufacture': None,
                'pinned': False
            }
        )

    def test_retrieve_v3(self):
        response = self._retrieve({'pk': 1, 'version': 'v3'})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data,
            {
                "id": 1,
                "number": "01A123AA",
                "tech_pass_num": "123454",
                "tech_pass_series": "AAA",
                "mark": "Tesla1",
                "name": "Krutaya tachka1",
                'image': None,
                "confidant": {
                    "id": 4,
                    "kind": "special",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01",
                    "issued_to": "First, Second, Third"
                },
                "inspection": {
                    "id": 4,
                    "car_detail": 1,
                    "dateInpsection": "2020-01-01T00:00:00.100000+05:00",
                    "dateNextInpsection": None,
                    "pDivision": "Toshkent"
                },
                "insurance": {
                    "id": 4,
                    "kind": "osago",
                    "number_of_drivers": 1,
                    "company_name": "insurance_company",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01"
                },
                "tinting": {
                    "id": 4,
                    "kind": "back",
                    "series": "ART",
                    "number": "12345678910",
                    "date_passed": "2017-01-01",
                    "date_end": "2040-01-01"
                },
                "oil": {
                    "id": 4,
                    "mark": "Castrol",
                    "mileage": 14000,
                    "next_mileage": 24000,
                    "charge_price": 1000000,
                    "last_replacement": "2020-01-01",
                    "next_replacement": "2023-01-01",
                    "daily_mileage": None,
                    "oil_mileage": None
                },
                'car_model': {
                    'id': 1,
                    "car_brand": 1,
                    'name': 'Nexia',
                    'image': {
                        'id': 1,
                        'name': 'my_file',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                'car_brand': {
                    'id': 1,
                    'name': 'Chevrolet',
                    'logo': {
                        'id': 1,
                        'name': 'my_file',
                        'file': 'http://testserver/media/test.jpg'
                    }
                },
                "fines_count": 2,
                "fines_remain": 10001000.00,
                "car_detail": 1,
                'date_of_registration': None,
                'year_manufacture': None,
                'pinned': False
            }
        )

    @patch('apps.cars.tasks.unsubscribe_to_fines')
    def test_task_unsubscribe_cars(self, mock_unsubscribe):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': 0}
        mock_unsubscribe.return_value = mock

        for pk in [6, 5]:
            car_detail = CarDetail.raw_objects.get(id=pk)
            self.assertTrue(car_detail.is_deleted)
            self.assertFalse(car_detail.is_subscribed)
            self.assertTrue(car_detail.require_unsubscribe)

        unsubscribe_cars()
        mock_unsubscribe.assert_has_calls([
            call(CarDetail.raw_objects.get(id=6)),
            call().json(),
            call(CarDetail.raw_objects.get(id=5)),
            call().json(),
        ])

        for pk in [6, 5]:
            car_detail = CarDetail.raw_objects.get(id=pk)
            self.assertTrue(car_detail.is_deleted)
            self.assertFalse(car_detail.is_subscribed)
            self.assertFalse(car_detail.require_unsubscribe)

    def test_manual_delete_car_detail(self):
        self.assertFalse(CarDetail.objects.get(id=1).is_deleted)
        CarDetail.objects.get(id=1).delete()
        self.assertTrue(CarDetail.raw_objects.get(id=1).is_deleted)

    def test_manual_delete_car(self):
        self.assertFalse(Car.objects.get(id=1).is_deleted)
        self.assertFalse(Car.objects.get(id=1).car_detail.is_deleted)

        Car.objects.get(id=1).delete()
        self.assertTrue(Car.raw_objects.get(id=1).is_deleted)
        self.assertFalse(Car.raw_objects.get(id=1).car_detail.is_deleted)

    @patch('apps.cars.tasks.unsubscribe_to_fines')
    def test_task_unsubscribe_cars_code_fail(self, mock_unsubscribe):
        mock = Mock()
        mock.status_code = 400
        mock.json.return_value = {'code': -100}
        mock_unsubscribe.return_value = mock

        CarDetail.objects.filter(id=1).update(is_subscribed=False)
        unsubscribe_cars()

        car_detail = CarDetail.raw_objects.get(id=1)
        self.assertFalse(car_detail.is_deleted)

    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create_already_created_car(self, mock_subscribe_to_fines):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': 0}
        mock_subscribe_to_fines.return_value = mock

        car_detail = CarDetail.raw_objects.get(id=6)
        self.assertTrue(car_detail.is_deleted)
        self.assertTrue(car_detail.require_unsubscribe)
        self.assertFalse(car_detail.is_subscribed)

        self.assertEqual(CarDetail.objects.count(), 4)
        data = {
            "number": "01J666DB",
            "tech_pass_series": "AAF",
            "tech_pass_num": "0476916",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(CarDetail.objects.count(), 5)

        car_detail = CarDetail.raw_objects.get(id=6)
        self.assertFalse(car_detail.is_deleted)
        self.assertFalse(car_detail.require_unsubscribe)
        self.assertTrue(car_detail.is_subscribed)

        mock_subscribe_to_fines.assert_called_once()
        mock_subscribe_to_fines.assert_called_with(car_detail)

    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create_fail_code_minus_100(self, mock_subscribe_to_fines):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': -100}
        mock_subscribe_to_fines.return_value = mock

        data = {
            "number": "01A127AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'non_field_errors': [
                    gettext(
                        'Error saferoad registration: %s'
                    ) % TEMPORARLY_UNAVAILABLE
                ]
            }
        )
        car_detail = CarDetail.raw_objects.get(
            number=data['number'],
            tech_pass_series=data['tech_pass_series'],
            tech_pass_num=data['tech_pass_num']
        )
        self.assertFalse(car_detail.is_deleted)
        self.assertFalse(car_detail.is_subscribed)
        self.assertTrue(car_detail.require_unsubscribe)

    @patch('apps.cars.services.cars.subscribe_to_fines')
    def test_create_fail_code_minus_4(self, mock_subscribe_to_fines):
        mock = Mock()
        mock.status_code = 200
        mock.json.return_value = {'code': -4, 'message': 'bbbb'}
        mock_subscribe_to_fines.return_value = mock

        data = {
            "number": "01A127AA",
            "tech_pass_series": "AAA",
            "tech_pass_num": "231231",
            "mark": "Tesla2",
            "name": "Krutaya tachka2",
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {
                'non_field_errors': [
                    gettext('Error saferoad registration: %s') % 'bbbb'
                ]
            }
        )
        car_detail = CarDetail.raw_objects.get(
            number=data['number'],
            tech_pass_series=data['tech_pass_series'],
            tech_pass_num=data['tech_pass_num']
        )
        self.assertTrue(car_detail.is_deleted)
        self.assertFalse(car_detail.is_subscribed)
        self.assertFalse(car_detail.require_unsubscribe)
