# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages
from ..models import InspectionPlace


class InspectionPlaceTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'inspection_places.yaml'
    ]

    url = 'myadmin:inspection-places-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_non_admin(self):
        self._auth_non_admin()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_requerements_field(self):
        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "name": [messages.INVALID_REQUIRED]
        })

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 1,
            "name": "Mirobod"
        })

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        self.assertEqual(
            response.data['results'][0], {
                "id": 3,
                "name": "Yunusobod"
            })

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 2)

    def test_create(self):
        data = {
            "name": "OSGO bektemir",
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(response.data, {
            "id": response.data['id'],
            "name": "OSGO bektemir"
        })

    def test_update_partial(self):
        data = {
            "name": "OSGO bektemir",
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['name'],
            data['name']
        )

    def test_update(self):
        data = {
            "name": "OSGO bektemir"
        }
        response = self._update({'pk': 2}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 2,
            "name": "OSGO bektemir"
        })

    def test_destroy(self):
        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        place = InspectionPlace.raw_objects.get(id=1)
        self.assertEqual(place.is_deleted, True)
