# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages
from ..models import CarBrand


class CarBrandTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'car_brand.yaml',
        'file.yaml',
        'user.yaml'
    ]

    url = 'myadmin:car_brands-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_non_admin(self):
        self._auth_non_admin()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_requerements_field(self):
        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "name": [messages.INVALID_REQUIRED]
        })

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 1,
            "name": "Chevrolet",
            "logo": {
                'id': 1,
                'name': 'my_file',
                'file': 'http://testserver/media/test.jpg'
            }
        })

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        self.assertEqual(response.data['results'][0], {
            "id": 1,
            "name": "Chevrolet",
            "logo": {
                'id': 1,
                'name': 'my_file',
                'file': 'http://testserver/media/test.jpg'
            }
        })

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 2)

    def test_create(self):
        data = {
            "name": "OSGO bektemir",
            "logo": 2,
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(response.data, {
            "id": response.data['id'],
            "name": "OSGO bektemir",
            "logo": {
                'id': 2,
                'name': 'my_file2',
                'file': 'http://testserver/media/test.jpg'
            }
        })

    def test_update_partial(self):
        data = {
            "name": "OSGO bektemir",
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(
            response.data['name'],
            data['name']
        )

    def test_update(self):
        data = {
            "name": "OSGO bektemir",
            "logo": 3
        }
        response = self._update({'pk': 2}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 2,
            "name": "OSGO bektemir",
            "logo": {
                'id': 3,
                'name': 'my_file3',
                'file': 'http://testserver/media/test.jpg'
            }
        })

    def test_destroy(self):
        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        car_brand = CarBrand.raw_objects.get(id=1)
        self.assertEqual(car_brand.is_deleted, True)
