# Stdlib
from base64 import b64decode
import re
from datetime import datetime, timedelta
from hashlib import md5

# Django
from django.core.files.base import ContentFile
from django.db.models import F, ExpressionWrapper, DateField
from django.conf import settings
from django.utils import timezone

# Pypi: celery
from celery import shared_task

# Pypi: fitz
import fitz

# Pypi:BeautifulSoup
from bs4 import BeautifulSoup

# Pypi: dateutil
from dateutil.parser import parse

# Pypi: requests
from requests.exceptions import ReadTimeout

# Project
from apps.files.models import File
from apps.notifications.services import (
    send_fine_notification, send_inspections_end_notification
)
from apps.notifications.models import NotificationTemplate
from apps.shared.utils import plural_days, requests, memcache_lock
from .models import Fine, Inspection, CarDetail, Insurance, Tinting, Confidant
from .services.cars import unsubscribe_to_fines
from .services.inspection import get_date_of_registration


def _extract_images_from_pdf(pdf: bytes):
    pdf_file = fitz.open(stream=pdf, filetype='pdf')
    page = pdf_file[0]
    images = []

    for image_index, img in enumerate(page.getImageList(), start=1):
        xref = img[0]
        base_image = pdf_file.extractImage(xref)
        image_bytes = base_image["image"]
        image_ext = base_image["ext"]

        file_name = 'plate_number_car' if image_index else 'plate_number'
        file = File.objects.create(name=file_name)
        file.file.save('file_name.%s' % image_ext,
                       ContentFile(image_bytes))

        images.append(file)

    return images


def _extract_date_of_ruling_from_pdf(pdf: bytes):
    pdf_file = fitz.open(stream=pdf, filetype='pdf')
    page = pdf_file[0]

    for i in page.getTextBlocks()[3:5]:
        try:
            text_with_date = i[4]
            match = re.search(r'\d{2}.\d{2}.\d{4}', text_with_date)
            date_of_ruling = match.group()
            return parse(date_of_ruling).date()
        except Exception:
            pass


@shared_task
def task_parse_pdf_fine(fine):
    if type(fine) == int:
        fine = Fine.objects.get(id=fine)

    try:
        response = requests.get(fine.pUrl)
        response.raise_for_status()
    except Exception as e:
        return None

    soup = BeautifulSoup(response.text, 'html.parser')
    try:
        data = str(soup.findAll('script')[2])
    except Exception as e:
        fine.is_exist_pdf = False
        fine.save()
        return None

    m = re.search('var content = (.+)[,;]{1}', data)

    pdf = b64decode(m.group(1))

    # Save pdf
    file = File.objects.create(name='PDF_DYHXX')
    file.file.save('preview.pdf', ContentFile(pdf))
    fine.pdf = file

    pdf_images = _extract_images_from_pdf(pdf)
    fine.plate_number = pdf_images[0]
    fine.plate_number_and_car = pdf_images[1]
    fine.date_of_ruling = _extract_date_of_ruling_from_pdf(pdf)

    fine.save()


@shared_task
def task_notify_discount_ending(*args, **kwargs):
    discount_left_days = [2, 7]
    now_date = datetime.now().date()
    discount_dates = [now_date + timedelta(days=i) for i in discount_left_days]

    fines = Fine.objects.annotate(
        date_of_end_discount=ExpressionWrapper(
            F('date_of_ruling') + timedelta(days=settings.MAX_DAYS_DISCOUNT),
            output_field=DateField()
        )
    ).filter(
        parent__isnull=True,
        pStatus=1,
        date_of_end_discount__in=discount_dates
    )

    [send_fine_notification(
        fine,
        NotificationTemplate.FINE_DISCOUNT_ENDING,
        {
            'date_of_end_discount': fine.date_of_end_discount.date(),
            'day_left': (fine.date_of_end_discount.date() - now_date).days,
            'plural_days': plural_days(
                (fine.date_of_end_discount.date() - now_date).days
            )
        }
    ) for fine in fines]


@shared_task(bind=True)
def task_refresh_remain(self):
    """
    Execute task only one at a time
    """
    feed_url = 'task_refresh_remain'.encode('utf8')
    feed_url_hexdigest = md5(feed_url).hexdigest()
    lock_id = '{0}-lock-{1}'.format(self.name, feed_url_hexdigest)
    with memcache_lock(lock_id, self.app.oid) as acquired:
        if acquired:
            from apps.cars.services.fines import refresh_remain
            fines = Fine.objects.filter(
                parent__isnull=True,
                is_fail_http_request_remain=True
            )
            for fine in fines:
                try:
                    refresh_remain(fine)
                except AssertionError:
                    pass


@shared_task
def repeat_parse_pdf():
    for fine in Fine.objects.failed_fines()[:30]:
        task_parse_pdf_fine(fine)
        if fine.plate_number and fine.plate_number_and_car:
            send_fine_notification(
                fine,
                NotificationTemplate.FINE_IMAGES_FOUND
            )


@shared_task
def notify_inspection_ending():
    now = timezone.now().date()
    to_date = now + timedelta(days=13)
    inspections = Inspection.objects.filter(
        dateNextInpsection__date__range=[now, to_date]
    ).select_related('car_detail')
    send_inspections_end_notification(
        inspections,
        NotificationTemplate.INSPECTION_COMING_END
    )


@shared_task
def unsubscribe_cars():
    car_details = CarDetail.raw_objects.filter(require_unsubscribe=True)[:20]
    for car_detail in car_details:
        try:
            response = unsubscribe_to_fines(car_detail)
            code = response.json()['code']
            if code in [0, -10, -2, -5]:
                car_detail.require_unsubscribe = False
                car_detail.is_subscribed = False
                car_detail.is_deleted = True
                car_detail.save()
        except Exception as e:
            continue


@shared_task
def get_cars_date_of_registration():
    items = CarDetail.objects.filter(is_checked_tech_passport=False)[:40]
    for car_detail in items:
        get_date_of_registration(car_detail)


@shared_task
def notify_insurance_ending():
    now = timezone.now().date()
    to_date = now + timedelta(days=13)
    insurances = Insurance.objects.filter(
        date_end__date__range=[now, to_date]
    )
    send_inspections_end_notification(
        insurances,
        NotificationTemplate.INSURANCE_COMING_END
    )


@shared_task
def notify_confidant_ending():
    now = timezone.now().date()
    to_date = now + timedelta(days=13)
    confidants = Confidant.objects.filter(
        date_end__date__range=[now, to_date]
    )
    send_inspections_end_notification(
        confidants,
        NotificationTemplate.CONFIDANT_COMING_END
    )


@shared_task
def notify_tinting_ending():
    now = timezone.now().date()
    to_date = now + timedelta(days=13)
    tintings = Tinting.objects.filter(
        date_end__date__range=[now, to_date]
    )
    send_inspections_end_notification(
        tintings,
        NotificationTemplate.TINTING_COMING_END
    )
