# Stdlib
import os

# Django
from django.core.cache import cache

# Pypi: dateutil
from dateutil.parser import parse

# Project
from apps.shared.utils import requests
from ..models import Inspection, CarDetail

BASE_URL = 'https://api-dtp.yhxbb.uz/api/admin/v1/'


def _create_token():
    response = requests.post(BASE_URL + 'auth/api-token-auth/', json={
        "username": os.environ['DTP_LOGIN'],
        "password": os.environ['DTP_PASSWORD']
    })
    response.raise_for_status()
    token = response.json()['token']

    cache.set('inspection_token', token)


def _get_token():
    token = cache.get('inspection_token')
    if not token:
        _create_token()
        token = cache.get('inspection_token')
    return token


def _get_inspection(car_detail, token):
    return requests.post(
        BASE_URL + 'cards/receiving/inspection',
        json={
            "card": "17016",
            "tech_passport_series": car_detail.tech_pass_series,
            "tech_passport_number": car_detail.tech_pass_num,
            "plate_number": car_detail.number
        },
        headers={
            'Authorization': 'Token %s' % token
        }
    )


def check_inspection(car_detail: CarDetail) -> [bool, dict]:
    token = _get_token()
    response = _get_inspection(car_detail, token)
    if response.status_code == 401:
        cache.delete('inspection_token')
        token = _get_token()
        response = _get_inspection(car_detail, token)

    response.raise_for_status()
    response_json = response.json()
    status = response_json['result']['Code']

    if status == 0:
        data = response_json['result']['Vehicle'][0]
        dateInpsection = data['Inspection']['dateInpsection']
        dateNextInpsection = data['Inspection']['dateNextInpsection']

        if dateInpsection:
            default = dict(
                value=data,
                pDivision=data['pDivision']
            )
            if dateNextInpsection:
                default['dateNextInpsection'] = parse(dateNextInpsection)

            obj, created = Inspection.objects.get_or_create(
                dateInpsection=parse(dateInpsection),
                car_detail=car_detail,
                defaults=default
            )
            if created:
                return True, response_json
    return False, response_json


def _get_date_of_registration(car_detail: CarDetail, token):
    return requests.post(
        BASE_URL + 'cards/receiving/vehicle_info_by_tech_pass',
        json={
            "card": "17016",
            "tech_pass_series": car_detail.tech_pass_series,
            "tech_pass_number": car_detail.tech_pass_num,
        },
        headers={
            'Authorization': 'Token %s' % token
        }
    )


def get_date_of_registration(car_detail: CarDetail):
    token = _get_token()
    response = _get_date_of_registration(car_detail, token)

    if response.status_code == 401:
        cache.delete('inspection_token')
        token = _get_token()
        response = _get_date_of_registration(car_detail, token)

    response.raise_for_status()
    response_json = response.json()

    car_detail.date_of_registration = response_json.get('pDateRegistration')
    car_detail.year_manufacture = response_json.get('Year')
    car_detail.is_checked_tech_passport = True
    car_detail.save()
