# Django
from django.db import transaction
from django.conf import settings
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework import status

# Project
from apps.notifications.services import (
    send_fine_notification,
    NotificationTemplate
)
from apps.transactions.services import get_fine_check
from ..models import Fine
from ..tasks import task_parse_pdf_fine


def _create_child_fine(**data):
    parent = Fine.objects.filter(
        pSeryNumber=data['pSeryNumber'],
        parent__isnull=True,
    ).exclude(pStatus=Fine.CHILD_STATUS).first()

    obj = Fine.objects.create(parent=parent, **data)

    if parent:
        parent.refresh_from_db()
        if parent.pStatus != 5:
            send_fine_notification(obj, NotificationTemplate.PARTIAL_PAID_FINE)

        try:
            refresh_remain(parent)
        except Exception as e:
            pass

    return obj


def _create_parent_fine(data):
    data['remain'] = data['pAmount']
    pSeryNumber = data.pop('pSeryNumber')

    obj, created = Fine.objects.update_or_create(
        pSeryNumber=pSeryNumber,
        parent__isnull=True,
        pStatus__in=Fine.PARENT_STATUSES,
        defaults={**data, 'parent': None}
    )
    Fine.objects.filter(
        pStatus=Fine.CHILD_STATUS,
        pSeryNumber=pSeryNumber
    ).update(parent=obj)

    if created:
        send_fine_notification(obj, NotificationTemplate.NEW_FINE)

    if settings.ENV == 'production':
        if obj.pUrl and created:
            transaction.on_commit(lambda: task_parse_pdf_fine.delay(obj.id))

        try:
            refresh_remain(obj)
        except Exception as e:
            pass


def _pay_fine(data):
    fine = Fine.objects.filter(
        pSeryNumber=data['pSeryNumber'],
        parent__isnull=True,
        pStatus__in=Fine.PARENT_STATUSES
    ).first()

    if not fine:
        fine = Fine.objects.create(**data)

    old_status = fine.pStatus

    fine.pStatus = 5
    fine.remain = 0
    fine.save()

    if old_status != 5:
        send_fine_notification(fine, NotificationTemplate.COMPLETELY_PAID_FINE)


@transaction.atomic
def create_fine(**data):
    status = data['pStatus']
    data['pAmount'] = data['pAmount'] / 100

    if status == 4:
        _create_child_fine(**data)
    elif status == 5:
        _pay_fine(data)
    elif status in [2, 3]:
        Fine.objects.filter(
            pSeryNumber=data['pSeryNumber'],
            parent__isnull=True,
            pStatus__in=Fine.PARENT_STATUSES
        ).update(pStatus=status)
    else:
        _create_parent_fine(data)

    return {}


@transaction.atomic
def refresh_remain(fine: Fine):
    assert fine.parent is None
    is_paid = fine.pStatus == 5

    try:
        response = get_fine_check(fine.pSeryNumber)
        data = response.text
        assert response.status_code == status.HTTP_200_OK, data

        fine.is_fail_http_request_remain = False
    except Exception as e:
        fine.is_fail_http_request_remain = True
        fine.save()
        raise AssertionError(_("Не удалось проверить статус штрафа."))

    data = response.json()
    if not data['status']:
        message = data['error']['message']
        if 'полностью оплачено' in message and not is_paid:
            fine.remain = 0
            fine.pStatus = 5
    else:
        fine.remain = float(data['result']['payee']['amount']) / 100
    fine.save()
    return fine, data
