# Stdlib
import hashlib

# Django
from django.conf import settings
from django.db import IntegrityError
from django.utils.translation import gettext as _

# Django Rest Framework
from rest_framework import status
from rest_framework.validators import ValidationError

# Pypi: requests
from requests.exceptions import ReadTimeout

# Project
from apps.shared.utils import requests
from ..models import Car, CarDetail
from ..errors import TEMPORARLY_UNAVAILABLE
from ..exceptions import AsbtInternalError, AsbtValidationError


def _generate_outdata_fine(car_detail: CarDetail, pStatus):
    phone = '998998551223'
    sign_string = hashlib.md5(
        ('%s%s%s%s%s' % (
            settings.SAFE_ROAD['SIGN_KEY'],
            car_detail.tech_pass_series,
            car_detail.tech_pass_num,
            car_detail.number,
            phone
        )).encode('utf8')
    ).hexdigest()
    return {
        "pId": car_detail.id,
        "pTexpSery": car_detail.tech_pass_series,
        "pTexpNumber": car_detail.tech_pass_num,
        "pPlateNumber": car_detail.number,
        "pPhone": phone,
        "pStatus": pStatus,
        "pProvider": "road24",
        "pSignString": sign_string
    }


def subscribe_to_fines(car_detail: CarDetail):
    data = _generate_outdata_fine(car_detail, 1)
    return requests.post(settings.SAFE_ROAD['URL'], json=data)


def unsubscribe_to_fines(car_detail: CarDetail):
    data = _generate_outdata_fine(car_detail, 0)
    return requests.post(settings.SAFE_ROAD['URL'], json=data)


def create_car(data):
    data_car_detail = data.pop('car_detail')
    car_detail = CarDetail.raw_objects.filter(**data_car_detail).first()
    if not car_detail:
        car_detail = CarDetail.objects.create(**data_car_detail)
    data['car_detail'] = car_detail

    if not car_detail.is_subscribed:
        msg = _('Error saferoad registration: %s')
        try:
            response = subscribe_to_fines(car_detail)
            if response.status_code != status.HTTP_200_OK:
                raise AsbtInternalError("Saferoad down.")
            response = response.json()

            if response['code'] == -100:
                raise AsbtInternalError(msg % TEMPORARLY_UNAVAILABLE)

            if response['code'] in [-4, -2]:
                raise AsbtValidationError(msg % response['message'])

            if response['code'] != 0:
                raise AssertionError(msg % response['message'])

            car_detail.is_deleted = False
            car_detail.is_subscribed = True
            car_detail.require_unsubscribe = False
            car_detail.save()
        except ReadTimeout:
            car_detail.require_unsubscribe = True
            car_detail.save()
            raise AssertionError(msg % "Saferoad timeout")
        except AsbtValidationError as e:
            car_detail.delete()
            raise AssertionError(e)
        except (AssertionError, AsbtInternalError) as e:
            car_detail.require_unsubscribe = True
            car_detail.save()
            raise AssertionError(e)

    try:
        car = Car.objects.create(**data)
    except IntegrityError:
        raise ValidationError({'number': [_('Already exists.')]})

    return car


def update_car(car, data):
    data.pop('car_detail', None)

    Car.objects.filter(id=car.id).update(**data)
    return Car.objects.get(id=car.id)


def delete_car(car):
    car.delete()
