# Django Rest Framework
from rest_framework.permissions import IsAuthenticated
from rest_framework.viewsets import ReadOnlyModelViewSet
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework import status

# Pypi: requests
from requests.exceptions import HTTPError, ReadTimeout

# Project
from ...models import Inspection
from ...services.inspection import check_inspection
from ..serializers.inspection import (
    InspectionSerializer, InspectionCheckSerializer
)


class InspectionViewSet(ReadOnlyModelViewSet):
    permission_classes = (IsAuthenticated,)
    filter_fields = ('car_detail',)

    def get_queryset(self):
        request = self.request
        user_cars = self.request.user.cars.values_list('car_detail', flat=True)
        queryset = Inspection.objects.filter(car_detail__id__in=user_cars)
        if request.version == 'v1':
            queryset = queryset.filter(dateNextInpsection__isnull=False)

        return queryset.order_by('-dateInpsection')

    def get_serializer_class(self):
        if self.action == 'check':
            return InspectionCheckSerializer
        return InspectionSerializer

    @action(methods=['POST'], detail=False)
    def check(self, request, pk=None, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        car_detail = serializer.validated_data['car_detail']

        try:
            is_new, obj = check_inspection(car_detail)
        except (HTTPError, ReadTimeout):
            msg = "Сервер тех. осмотра временно не работает."
            return Response(msg, status=status.HTTP_503_SERVICE_UNAVAILABLE)

        return Response({
            'response': obj,
            'is_new': is_new
        })
