# Python
from collections import defaultdict, deque

# Django Rest Framework
from rest_framework.permissions import IsAuthenticated
from rest_framework.viewsets import ReadOnlyModelViewSet
from rest_framework.generics import ListAPIView
from rest_framework.decorators import action
from rest_framework.response import Response

# Project
from apps.transactions.models import Check
from ...models import Fine
from ...filters import FineFilterSet
from ..serializers.fine import (
    FineSerializer,
    FineListSerializer,
    PayFineSerializer,
    PaymentsListSerializer
)
from ...services.fines import refresh_remain


class FineViewSet(ReadOnlyModelViewSet):
    permission_classes = (IsAuthenticated,)
    filterset_class = FineFilterSet

    def get_queryset(self):
        return Fine.objects.filter(
            car_detail__cars__user=self.request.user,
            car_detail__cars__is_deleted=False,
            parent__isnull=True
        ).select_related('pViolation').distinct()

    def get_serializer_class(self):
        if self.action == 'list':
            return FineListSerializer
        if self.action == 'pay':
            return PayFineSerializer
        return FineSerializer

    @action(methods=['post'], detail=True)
    def pay(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return Response()

    @action(methods=['get'], detail=True)
    def refresh_remain(self, request, *args, **kwargs):
        instance = self.get_object()
        instance, data = refresh_remain(instance)
        return Response(data)


class PaymentsAPIView(ListAPIView):
    serializer_class = PaymentsListSerializer
    filterset_class = FineFilterSet

    def get_queryset(self):
        return Fine.objects.filter(
            car_detail__cars__user=self.request.user,
            car_detail__cars__is_deleted=False,
            parent__isnull=False
        ).select_related('pViolation', 'car_detail', 'parent').distinct()

    def filter_queryset(self, queryset):
        qs = super().filter_queryset(queryset)

        checks = Check.objects.filter(
            fine__car_detail__cars__user=self.request.user,
        ).values('id', 'fine_id', 'amount')
        check_ids = defaultdict(lambda: deque())
        [check_ids[(check['fine_id'], check['amount'] / 100)].append(
            check['id']) for check in checks]

        for fine in qs:
            if check_ids[(fine.parent_id, fine.pAmount)]:
                fine.check_id = check_ids[(fine.parent_id, fine.pAmount)].pop()

        return qs
