# Stdlib
from datetime import timedelta
from decimal import Decimal
from random import randint

# Django
from django.utils import timezone
from django.utils.translation import gettext as _
from django.conf import settings
from django.core.cache import cache

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.files.serializers import FileSerializer
from apps.bank_cards.models import BankCard
from apps.bank_cards.utils import get_bank_card_detail
from apps.shared.utils import send_sms
from apps.transactions.services import pay_fine
from apps.shared.serializers import DynamicFieldsModelSerializer
from .car_detail import CarDetailSerializer
from ...models import Fine, Violation, Car


class PaidFineSerializer(serializers.ModelSerializer):
    class Meta:
        model = Fine
        fields = (
            'id',
            'pAmount',
            'created_date'
        )


class ViolationSerializer(serializers.ModelSerializer):
    class Meta:
        model = Violation
        fields = ('id', 'desc', 'article', 'part')


class FineSerializer(DynamicFieldsModelSerializer):
    pays = PaidFineSerializer(source='child', many=True)
    pdf = FileSerializer()
    plate_number = FileSerializer()
    plate_number_and_car = FileSerializer()
    pViolation = ViolationSerializer()
    discount_and_paid = serializers.SerializerMethodField()
    car = serializers.SerializerMethodField()

    def get_discount_and_paid(self, obj: Fine):
        now_date = timezone.now().date()
        discount_percent = 0
        amount = obj.pAmount

        date_of_ruling = obj.date_of_ruling or obj.pDate.date()
        day_for_discount = date_of_ruling + timedelta(
            days=settings.MAX_DAYS_DISCOUNT
        )
        can_discount = obj.pSeryNumber.lower().startswith('r')

        if can_discount and day_for_discount >= now_date:
            discount_percent = 30
            amount -= (amount * Decimal(f'0.{discount_percent}'))

        paid = amount - obj.remain
        return {
            'discount': discount_percent,
            'paid': 0 if paid < 0 else paid
        }

    def get_car(self, obj: Fine):
        user = self.context['request'].user
        car = Car.objects.get(
            user=user,
            car_detail__fines=obj
        )

        return {
            'id': car.id
        }

    class Meta:
        model = Fine
        fields = (
            'id',
            'pSeryNumber',
            'pPlace',
            'pLocation',
            'pDate',
            'pViolation',
            'pAmount',
            'remain',
            'pStatus',
            'pUrl',
            'pays',
            'pdf',
            'plate_number',
            'plate_number_and_car',
            'discount_and_paid',
            'car',
            'date_of_ruling'
        )


class FineListSerializer(serializers.ModelSerializer):
    plate_number_and_car = FileSerializer()
    pViolation = ViolationSerializer()
    check_id = serializers.IntegerField(required=False)

    class Meta:
        model = Fine
        fields = (
            'id',
            'pSeryNumber',
            'pPlace',
            'pLocation',
            'pViolation',
            'pDate',
            'pStatus',
            'plate_number_and_car',
            'remain',
            'pAmount',
            'check_id'
        )


class PayFineSerializer(serializers.ModelSerializer):
    bank_card = serializers.PrimaryKeyRelatedField(
        queryset=BankCard.objects.filter(is_verified=True)
    )
    amount = serializers.FloatField(min_value=1000)
    sms_code = serializers.CharField(required=False)

    class Meta:
        model = Fine
        fields = ('id', 'bank_card', 'amount', 'sms_code')

    def validate_sms_code(self, sms_code):
        cache_sms_code = cache.get('fine_%d_%s' % (self.instance.id, sms_code))
        if cache_sms_code != sms_code:
            raise serializers.ValidationError(_("Wrong sms code."))
        return sms_code

    def validate_bank_card(self, bank_card: BankCard):
        request = self.context['request']
        amount = self.initial_data['amount']
        user = request.user
        language = request.LANGUAGE_CODE

        if user != bank_card.user:
            raise serializers.ValidationError('Not found.')

        is_success, data = get_bank_card_detail(
            bank_card.number,
            bank_card.expire
        )
        if not is_success:
            lang_error = 'ru' if language == 'ru' else 'uz'
            raise serializers.ValidationError(
                data[lang_error]
            )

        if (data['balance'] / 100) < amount:
            msg = _('Not enough money on card.')
            raise serializers.ValidationError(msg)

        return bank_card

    def update(self, instance, validated_data):
        version = self.context['request'].version
        bank_card = validated_data['bank_card']
        amount = validated_data['amount']
        sms_code = validated_data.get('sms_code')

        if version == 'v2' and amount >= 50000 and not sms_code:
            sms_code = str(randint(100000, 999999))
            msg = _('Road24 dlya podtverjdeniya plateja '
                    'SMS-kod: %s %s') % (settings.SMS_APP_CODE, sms_code)

            cache.set('fine_%d_%s' % (instance.id, sms_code), sms_code, 60)
            send_sms(bank_card.phone, msg)
            return validated_data
        try:
            if sms_code:
                cache.delete('fine_%d_%s' % (instance.id, sms_code))
            pay_fine(bank_card, amount, instance)
        except AssertionError as e:
            raise serializers.ValidationError({
                'non_field_errors': [e]
            })
        return validated_data

    def create(self, validated_data):
        raise NotImplemented


class PaymentsListSerializer(serializers.ModelSerializer):
    pViolation = ViolationSerializer()
    check_id = serializers.IntegerField(required=False)
    car_detail = CarDetailSerializer()
    remain = serializers.DecimalField(
        source='parent.remain',
        max_digits=11,
        decimal_places=2
    )

    class Meta:
        model = Fine
        fields = (
            'id',
            'car_detail',
            'pSeryNumber',
            'pPlace',
            'pLocation',
            'pViolation',
            'pDate',
            'pStatus',
            'remain',
            'pAmount',
            'parent_id',
            'check_id'
        )
