# Django
from django.db import transaction
from django.utils.translation import ugettext as _

# Django Rest Framework
from rest_framework import serializers

# Project
from apps.files.serializers import FileSerializer
from .confidant import ConfidantSerializer
from .inspection import OldInspectionSerializer, InspectionSerializer
from .insurance import InsuranceSerializer
from .tinting import TintingSerializer
from .oil import OilSerializer
from .handbooks import CarBrandSerializer, CarModelSerializer
from ...models import Car, Confidant, OldInspection, Insurance, Tinting, Oil
from ...services.cars import create_car, update_car
from apps.shared.serializers import DynamicFieldsModelSerializer


class CarSerializer(DynamicFieldsModelSerializer):
    confidant = ConfidantSerializer(required=False)
    inspection = OldInspectionSerializer(required=False)
    insurance = InsuranceSerializer(required=False)
    tinting = TintingSerializer(required=False)
    oil = OilSerializer(required=False)

    fines_remain = serializers.ReadOnlyField()
    fines_count = serializers.IntegerField(read_only=True)

    number = serializers.CharField(source='car_detail.number')
    tech_pass_num = serializers.CharField(
        source='car_detail.tech_pass_num',
        max_length=10
    )
    tech_pass_series = serializers.CharField(
        source='car_detail.tech_pass_series',
        max_length=3
    )
    date_of_registration = serializers.DateTimeField(
        source='car_detail.date_of_registration', read_only=True)
    year_manufacture = serializers.IntegerField(
        source='car_detail.year_manufacture', read_only=True)

    def to_representation(self, instance):
        if self.fields.get('inspection'):
            self.fields['inspection'] = serializers.SerializerMethodField()
        if self.fields.get('car_brand'):
            self.fields['car_brand'] = CarBrandSerializer()
        if self.fields.get('car_model'):
            self.fields['car_model'] = CarModelSerializer()
        if self.fields.get('image'):
            self.fields['image'] = FileSerializer(
                fields=('id', 'name', 'file'))
        return super().to_representation(instance)

    def get_inspection(self, obj: Car):
        version = self.context['request'].version
        if version in ['v2', 'v3']:
            inspection = obj.car_detail.inspections.order_by(
                '-dateInpsection'
            )
            if version == 'v2':
                inspection = inspection.filter(
                    dateNextInpsection__isnull=False
                )

            inspection = inspection.first()
            if inspection:
                return InspectionSerializer(inspection).data
            return None
        if inspection := obj.inspection:
            return OldInspectionSerializer(inspection).data
        return None

    class Meta:
        model = Car
        fields = (
            'id',
            'number',
            'tech_pass_num',
            'tech_pass_series',
            'mark',
            'name',
            'image',
            'confidant',
            'inspection',
            'insurance',
            'tinting',
            'oil',
            'car_model',
            'car_brand',
            'fines_count',
            'fines_remain',
            'car_detail',
            'date_of_registration',
            'year_manufacture',
            'pinned'
        )
        read_only_fields = ('car_detail',)

    def nested_update_or_create(self, name, model, validated_data):
        if obj_data := validated_data.pop(name, None):
            if obj := getattr(self.instance, name):
                obj_pk = obj.pk
                model.objects.filter(id=obj_pk).update(**obj_data)
            else:
                model.objects.create(car=self.instance, **obj_data)

    @transaction.atomic
    def update(self, instance, validated_data):
        self.nested_update_or_create('confidant', Confidant, validated_data)
        self.nested_update_or_create(
            'inspection', OldInspection, validated_data
        )
        self.nested_update_or_create('insurance', Insurance, validated_data)
        self.nested_update_or_create('tinting', Tinting, validated_data)
        self.nested_update_or_create('oil', Oil, validated_data)

        try:
            instance = update_car(instance, validated_data)
        except AssertionError as e:
            raise serializers.ValidationError({'non_field_errors': [e]})
        return instance

    def create(self, validated_data):
        validated_data['user'] = self.context['request'].user
        confidant = validated_data.pop('confidant', None)
        inspection = validated_data.pop('inspection', None)
        insurance = validated_data.pop('insurance', None)
        tinting = validated_data.pop('tinting', None)
        oil = validated_data.pop('oil', None)

        try:
            instance = create_car(validated_data)
        except AssertionError as e:
            raise serializers.ValidationError({'non_field_errors': [e]})
        except serializers.ValidationError as e:
            raise e

        if confidant:
            Confidant.objects.create(car=instance, **confidant)

        if inspection:
            OldInspection.objects.create(car=instance, **inspection)

        if insurance:
            Insurance.objects.create(car=instance, **insurance)

        if tinting:
            Tinting.objects.create(car=instance, **tinting)

        if oil:
            Oil.objects.create(car=instance, **oil)

        return instance
